<?php

namespace Marcolin\Models;

use Illuminate\Database\Eloquent\Builder;

/**
 * Class Promotion
 * @package Hilco\Models
 *
 * @method static Builder active()
 */
class PromotionCurrentOrderValueTrigger extends PromotionTriggerDetails {
    protected $table = 'PromotionCurrentOrderValueTriggers';
    protected $fillable = ['target_all_products', 'minimum_value', 'almost_qualified', 'include_backorders', 'commit_sequence'];

    // Overrides Model::boot(), used to define a specific 'deleted' state
    public static function boot() {
        parent::boot();
        // make sure all related rows in PromotionCurrentOrderValueTrigger_Product are also deleted
        static::deleted(function (PromotionCurrentOrderValueTrigger $promotionCurrentOrderValueTrigger) {
            $promotionCurrentOrderValueTriggerProducts =
                PromotionCurrentOrderValueTrigger_Product
                    ::where('promotioncurrentordervaluetrigger_id', '=', $promotionCurrentOrderValueTrigger->id)
                    ->get();

            foreach ($promotionCurrentOrderValueTriggerProducts as $promotionCurrentOrderValueTriggerProduct) {
                $promotionCurrentOrderValueTriggerProduct->delete();
            }
        });
    }

    /**
     * Boolean, all promo trigger types need to implement to use their specific logic
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return mixed
     */
    public function isTriggered(Customer $soldToCustomer, $webCartItems = []) {
        $totalItemValue = $this->getCurrentQualifyingCartTotal($soldToCustomer, $webCartItems);

        return $totalItemValue >= $this->minimum_value;
    }

    /**
     * promo trigger types should implement if there post operations that should happen after promo is applied
     * @param $soldToCustomer
     * @return mixed
     */
    public function postTriggerOperation(Customer $soldToCustomer) {
        // TODO: Implement postTriggerOperation() method.
    }

    /**
     * Boolean, any promo trigger that can use the Almost Qualifying Header should implement this and return 'true'
     * otherwise return 'false'
     * @return bool
     */
    public function hasAlmostQualifyingField() {
        return true;
    }

    /**
     * Any promo trigger that can use the Almost Qualifying Header should implement this and
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return mixed
     */
    public function isAlmostQualifying(Customer $soldToCustomer, $webCartItems) {
        $totalItemValue = $this->getCurrentQualifyingCartTotal($soldToCustomer, $webCartItems);

        return $totalItemValue < $this->minimum_value;
    }

    /**
     * Get all WebBrands specifically targeted by this trigger
     * @return mixed
     */
    public function webBrands() {
        return $this
            ->morphedByMany(WebBrand::class, 'product',
                'PromotionCurrentOrderValueTrigger_Product', 'promotioncurrentordervaluetrigger_id')
            ->where('PromotionCurrentOrderValueTrigger_Product.deleted_at', '=', '0000-00-00 00:00:00')
            ;
    }

    /**
     * Get all WebStyles specifically targeted by this trigger
     * @return mixed
     */
    public function webStyles() {
        return $this
            ->morphedByMany(WebStyle::class, 'product',
                'PromotionCurrentOrderValueTrigger_Product', 'promotioncurrentordervaluetrigger_id')
            ->where('PromotionCurrentOrderValueTrigger_Product.deleted_at', '=', '0000-00-00 00:00:00')
            ;
    }

    /**
     * Get all WebUnits specifically targeted by this trigger
     * @return mixed
     */
    public function webUnits() {
        return $this
            ->morphedByMany(WebUnit::class, 'product',
                'PromotionCurrentOrderValueTrigger_Product', 'promotioncurrentordervaluetrigger_id')
            ->where('PromotionCurrentOrderValueTrigger_Product.deleted_at', '=', '0000-00-00 00:00:00')
            ;
    }

    /**
     * Gets the total value of 'qualifying' items in the cart, where 'qualifying' is defined as
     * - contained in the list of targeted web products (units, styles, brands), if not targeting all products
     * @param Customer $customer
     * @param $webCartItems
     * @return int
     */
    public function getCurrentQualifyingCartTotal(Customer $customer, $webCartItems) {
        $totalItemValue = 0;

        if (isset($webCartItems) && count($webCartItems) > 0) {
            if ($this->target_all_products) {
                foreach ($webCartItems as $webCartItem) {
                    if ($this->include_backorders || !$webCartItem->unit->isBackOrdered()) {
                        $totalItemValue = $totalItemValue + $webCartItem->getExtendedPrice($customer);
                    }
                }
            } else {
                $relevantUnits = $this->webUnits;
                $relevantStyles = $this->webStyles;
                $relevantBrands = $this->webBrands;

                foreach ($webCartItems as $webCartItem) {
                    $webUnit = $webCartItem->webUnit;
                    $webStyle = $webUnit->webStyle;
                    $webBrand = $webStyle->webBrand;
                    if ($this->include_backorders || !$webCartItem->unit->isBackOrdered()) {
                        if ((isset($webUnit) && $relevantUnits->contains($webUnit)) ||
                            (isset($webStyle) && $relevantStyles->contains($webStyle)) ||
                            (isset($webBrand) && $relevantBrands->contains($webBrand))) {
                            $totalItemValue = $totalItemValue + $webCartItem->getExtendedPrice($customer);
                        }
                    }
                }
            }
        }

        return $totalItemValue;
    }

}
