<?php

namespace Marcolin\Models;

use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Input;
use Venturecraft\Revisionable\RevisionableTrait;
use Visionware\DataManager\HasCustomLoggingTrait;

class WebBrand extends WebModel implements HasLandingPageInterface
{
    protected $table = "WebBrands";
    protected $fillable = ['name', 'slug', 'brand_id', 'is_visible', 'detail'];
    protected $casts = [
        ['is_visible' => 'boolean']
    ];
    protected $perPage = 24;

    public function getPerPage()
    {
        $tmpPerPage = Input::get('perPage');
        return (isset($tmpPerPage) && $tmpPerPage > 0) ? $tmpPerPage : $this->perPage;
    }

    public static $perEnvironment = true;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'name',
            'brand.brandCode',
            'brand.brandName'
        ],
        'attributesForFaceting' => [
        ]
    ];

    public function indexOnly()
    {
        return true;
    }

    public function getAlgoliaRecord()
    {
        $record = array_only($this->toArray(), [
            'id',
        ]);

        $record['name'] = $this->name;
        $record['slug'] = $this->slug;
        $record['visibility'] = [];
        if ($this->isWebVisibleStandard) {
            $record['visibility'][] = 'standard';
        }
        if ($this->isWebVisibleHarleyDavidson) {
            $record['visibility'][] = 'harley';
        }
        if ($this->isWebVisibleAutocat) {
            $record['visibility'][] = 'autocat';
        }
        if ($this->isWebVisibleTomFordLimited) {
            $record['visibility'][] = 'tom_ford_limited';
        }

        $record['brand'] = $this->brand;

        $unwanted_keys = ['date_created', 'date_modified', 'date_uploaded', 'deleted_at', 'commit_sequence'];
        $this->recursive_unset($record, $unwanted_keys);

        return $record;
    }

    function recursive_unset(&$array, $unwanted_keys)
    {
        foreach ($unwanted_keys as $unwanted_key) {
            unset($array[$unwanted_key]);
        }
        foreach ($array as &$value) {
            if (is_array($value)) {
                $this->recursive_unset($value, $unwanted_keys);
            }
        }
    }

    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];

    public static $elasticFuzziness = 1;

    /**
     * @description return elastic query with fuzziness
     */
    public static function buildElasticQuery($keyWords)
    {
        $params = [
            'multi_match' => [
                'query' => $keyWords,
                'fuzziness' => self::$elasticFuzziness,
                'fields' => ['_all']
            ]
        ];
        return $params;
    }

    public function webSilos()
    {
        return $this->belongsToMany(WebSilo::class, 'WebBrand_WebSilo', 'webbrand_id', 'websilo_id')
            ->withPivot('sort_name')
            ->orderBy('sort_name');
    }

    public function brand()
    {
        return $this->belongsTo(Brand::class, 'brand_id', 'id');
    }

    public function webBrand_Algolia()
    {
        return $this->hasMany(WebBrand_Algolia::class, 'webBrand_id', 'id');
    }

    public function algoliaTable()
    {
        return $this->hasMany(WebBrand_Algolia::class, 'webBrand_id', 'id');
    }

    public function getAlgoliaClassAttribute() {
        return WebBrand_Algolia::class;
    }

    public function getAlgoliaIDFieldAttribute() {
        return 'webBrand_id';
    }

    public function toString() {
        return $this->slug;
    }

    public function subBrands()
    {
        return $this->belongsToMany(WebBrand::class, 'WebBrand_SubWebBrand', 'parent_brand_id', 'sub_brand_id')->with('brand');
    }

    public function webStyles()
    {
        return $this->hasMany(WebStyle::class, 'webbrand_id', 'id');
    }

    public function scopeAlphabetical(Builder $query)
    {
        return $query->orderBy('name', 'asc');
    }

    public function getFeaturedChildrenAttribute()
    {
        return $this->visibleWebStyles()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute()
    {
        return $this->visibleWebStyles()->get();
    }


    public function scopeForCustomer($query, $customer = false)
    {
        return $query->whereHas('brand', function ($query) use ($customer) {
            $query->forCustomer($customer);
            return $query;
        });
    }

    public function scopePpe(Builder $query)
    {
        return $query
            ->where('is_ppe', '=', '1');
    }

    public function setWebCategoryIdsAttribute($ids)
    {
        $this->save();
        $this->createRevisionRecord('webCategoryIds', $this->webCategories->pluck('id')->toJson(), json_encode($ids));
        $this->webCategories()->sync($ids);
    }

    public function getHierarchyPathsAttribute()
    {
        $paths = [];
        foreach ($this->webCategories as $webCategory) {
            foreach ($webCategory->webBrands as $webBrand) {
                foreach ($webBrand->webSilos as $webSilo) {
                    $paths[] = [
                        'webSilo' => $webSilo,
                        'webBrand' => $webBrand,
                        'webCategory' => $webCategory,
                    ];
                }
            }
        }
        return $paths;
    }

    public function scopeInCollection($query, $webCollection)
    {
        return $query->whereHas('webBrands', function ($query) use ($webCollection) {
            $query->where('webcollection_id', $webCollection->id);
        });
    }

    public function scopeInSilo($query, $webSiloId)
    {
        return $query->whereHas('webCategories', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }

    public function getAssetPrefix()
    {
        return "webBrands/{$this->id}";
    }

    public function getIsWebVisibleAttribute()
    {
        return static::visible()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleAllAttribute()
    {
        return static::webVisibleAll()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleAutocatAttribute()
    {
        return static::webVisibleAutocat()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleStandardAttribute()
    {
        return static::webVisibleStandard()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleHarleyDavidsonAttribute()
    {
        return static::webVisibleHarleyDavidson()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleTomFordLimitedAttribute()
    {
        return static::webVisibleTomFordLimited()->where('id', '=', $this->id)->count() > 0;
    }

    public function scopeVisible($query)
    {
        $query
            ->whereHas('brand', function ($query) {
                return $query->webVisible();
            });
        return $query;
    }

    public function scopeWebVisibleAll($query)
    {
        $query
            ->whereHas('brand', function ($query) {
                return $query->webVisibleAll();
            });
        return $query;
    }

    public function scopeWebVisibleAutocat($query)
    {
        $query
            ->whereHas('brand', function ($query) {
                return $query->webVisibleAutocat();
            });
        return $query;
    }

    public function scopeWebVisibleStandard($query)
    {
        $query
            ->whereHas('brand', function ($query) {
                return $query->webVisibleStandard();
            });
        return $query;
    }

    public function scopeWebVisibleHarleyDavidson($query)
    {
        $query
            ->whereHas('brand', function ($query) {
                return $query->webVisibleHarleyDavidson();
            });
        return $query;
    }

    public function scopeWebVisibleTomFordLimited($query)
    {
        $query
            ->whereHas('brand', function ($query) {
                return $query->webVisibleHarleyDavidson();
            });
        return $query;
    }

    public function scopeForDisplay(Builder $query)
    {
        return $query
            ->where('WebBrands.is_sub_brand', '0');
    }

    public function slugUrl()
    {
        return route('collection.slug', [$this->slug]);
    }

    public function getParentRelationName()
    {
        return null;
    }

    public function getNameAttribute($name)
    {
        if (isset($name) && strlen($name) > 0) {
            return $name;
        }

        $name = $this->brand->name;
        if (isset($name) && strlen($name) > 0) {
            return $name;
        }

        return $this->slug;
    }

    public function getDisplayNameForUACAttribute() {
        return $this->name;
    }

    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait, ManyToManyRevisionableTrait;
}
