<?php
namespace Marcolin\Models\Enums;

use Marcolin\Models\SalesRep;
use Marcolin\Models\SalesTarget;
use Symfony\Component\HttpFoundation\AcceptHeader;

final class SalesTargetTypes extends BasicEnum {
    public static function SALES_DOLLARS() { return self::_create("SalesDollars"); }
    public static function ACCOUNTS_SOLD() { return self::_create("AccountsSold"); }
    public static function NEW_DOORS_OPENED() { return self::_create("NewDoorsOpened"); }
    public static function BRANDS_PER_DOOR() { return self::_create("BrandsPerDoor"); }
    public static function RETURN_RATE() { return self::_create("ReturnRate"); }

    public static function ALL_VALUES() {
        return [self::SALES_DOLLARS(), self::ACCOUNTS_SOLD(), self::NEW_DOORS_OPENED(), self::BRANDS_PER_DOOR(), self::RETURN_RATE()];
    }

    public function getDefaultValue() {
        switch ($this) {
            case SalesTargetTypes::RETURN_RATE(): {
                return 50;
            }
            case SalesTargetTypes::BRANDS_PER_DOOR(): {
                return 1.5;
            }
            case SalesTargetTypes::NEW_DOORS_OPENED(): {
                return 50;
            }
            case SalesTargetTypes::ACCOUNTS_SOLD(): {
                return 70;
            }
            case SalesTargetTypes::SALES_DOLLARS(): {
                return 5000;
            }
        }

        return 0;
    }

    public function getFormatType() {
        switch ($this) {
            case SalesTargetTypes::RETURN_RATE(): {
                return 'Percentage';
            }
            case SalesTargetTypes::BRANDS_PER_DOOR(): {
                return null;
            }
            case SalesTargetTypes::NEW_DOORS_OPENED(): {
                return null;
            }
            case SalesTargetTypes::ACCOUNTS_SOLD(): {
                return 'Percentage';
            }
            case SalesTargetTypes::SALES_DOLLARS(): {
                return 'Currency';
            }
        }

        return null;
    }

    public function getTitle() {
        switch ($this) {
            case SalesTargetTypes::RETURN_RATE(): {
                return 'Return Rate (Last 12 Mon)';
            }
            case SalesTargetTypes::BRANDS_PER_DOOR(): {
                return 'Brands/Door (Last 12 Mon)';
            }
            case SalesTargetTypes::NEW_DOORS_OPENED(): {
                return 'New Doors (Last 12 Mon)';
            }
            case SalesTargetTypes::ACCOUNTS_SOLD(): {
                return 'Acct. Pen. % (Last 12 Mon)';
            }
            case SalesTargetTypes::SALES_DOLLARS(): {
                return 'Sales Dollars (YTD)';
            }
        }

        return '';
    }

    public function getSalesTarget(SalesRep $salesRep) {
        return SalesTarget::salesTargetType($this)->forRep($salesRep)->first();
    }

    public function getTargetValue($salesRep) {
        $salesTarget = $this->getSalesTarget($salesRep);
        if ($salesTarget) {
            return $salesTarget->targetValue;
        } else {
            return null;
        }
    }

    public function isPassedTarget($salesRep) {
        $targetValue = $this->getTargetValue($salesRep);
        $currentValue = $this->getCurrentValue($salesRep);
        $defaultValue = $this->getDefaultValue();

        return ($targetValue != null && $currentValue >= $targetValue) || ($targetValue == null && $currentValue > $defaultValue);
    }

    public function getCurrentValue($salesRep) {
        return 100;
    }
}