<?php

namespace Marcolin\Models;
use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use App;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Input;
use Venturecraft\Revisionable\RevisionableTrait;

class WebBrand extends WebModel implements HasLandingPageInterface {
    protected $table = "WebBrands";
    protected $fillable = ['name', 'slug', 'brand_id', 'is_visible', 'detail'];
    protected $casts = [
        ['is_visible' => 'boolean']
    ];
    protected $perPage = 24;

    public function getPerPage()
    {
        $tmpPerPage =Input::get('perPage');
        return (isset($tmpPerPage) && $tmpPerPage > 0)?$tmpPerPage:$this->perPage;
    }

    use AlgoliaEloquentTrait;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'name',
            'brand.brandCode',
            'brand.brandName'
        ],
        'attributesForFaceting' => [
        ]
    ];

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

        static::visible()
            ->has('webSilos')
            ->has('brand')

            ->with('webSilos')
            ->with('brand')

            ->chunk(1000, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
                /** @var \AlgoliaSearch\Index $index */
                foreach ($indicesTmp as $index) {
                    $records         = [];
                    $recordsAsEntity = [];

                    foreach ($models as $model) {
                        if ($modelHelper->indexOnly($model, $index->indexName)) {
                            $records[] = $model->getAlgoliaRecordDefault($index->indexName);

                            if ($onInsert && is_callable($onInsert)) {
                                $recordsAsEntity[] = $model;
                            }
                        }
                    }

                    $index->addObjects($records);

                    if ($onInsert && is_callable($onInsert)) {
                        call_user_func_array($onInsert, [$recordsAsEntity]);
                    }
                }

            })
        ;

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }
    }

    public function indexOnly() {
        if (!$this->is_visible) return false;

        return true;
    }

    public function getAlgoliaRecord() {
        $record = array_only($this->toArray(), [
            'id',
        ]);

        $record['name'] = $this->name;
        $record['slug'] = $this->slug;

        $record['brand'] = $this->brand;

        return $record;
    }

    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];

    public static $elasticFuzziness = 1;

    /**
     * @description return elastic query with fuzziness
     */
    public static function buildElasticQuery($keyWords){
        $params=[
            'multi_match' => [
                'query'     => $keyWords,
                'fuzziness' => self::$elasticFuzziness,
                'fields'    => ['_all']
            ]
        ];
        return $params;
    }

    public function webSilos() {
        return $this->belongsToMany(WebSilo::class, 'WebBrand_WebSilo', 'webbrand_id', 'websilo_id')
            ->withPivot('sort_name')
            ->orderBy('sort_name')
            ;
    }

    public function brand() {
        return $this->belongsTo(Brand::class, 'brand_id', 'id');
    }

    public function subBrands() {
        return $this->belongsToMany(WebBrand::class, 'WebBrand_SubWebBrand', 'parent_brand_id', 'sub_brand_id')->with('brand');
    }

    public function webStyles() {
        return $this->hasMany(WebStyle::class, 'webbrand_id', 'id');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

    public function getFeaturedChildrenAttribute() {
        return $this->visibleWebStyles()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute() {
        return $this->visibleWebStyles()->get();
    }


    public function scopeForCustomer($query, $customer = false) {
        return $query->whereHas('brand', function ($query) use ($customer) {
            $query->forCustomer($customer);
            return $query;
        });
    }

    public function setWebCategoryIdsAttribute($ids) {
        $this->save();
        $this->createRevisionRecord('webCategoryIds', $this->webCategories->pluck('id')->toJson(), json_encode($ids));
        $this->webCategories()->sync($ids);
    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webCategories as $webCategory) {
            foreach ($webCategory->webBrands as $webBrand) {
                foreach ($webBrand->webSilos as $webSilo) {
                    $paths[] = [
                        'webSilo' => $webSilo,
                        'webBrand' => $webBrand,
                        'webCategory' => $webCategory,
                    ];
                }
            }
        }
        return $paths;
    }

    public function scopeInCollection($query, $webCollection) {
        return $query->whereHas('webBrands', function ($query) use ($webCollection) {
            $query->where('webcollection_id', $webCollection->id);
        });
    }

    public function scopeInSilo($query, $webSiloId) {
        return $query->whereHas('webCategories', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }    

    public function getAssetPrefix() {
        return "webBrands/{$this->id}";
    }

    public function scopeVisible(Builder $query) {
        return $query
            ->whereHas('brand', function ($query) {
                return $query->webVisible();
            })
            ;
    }

    public function scopeForDisplay(Builder $query) {
        return $query
            ->where('WebBrands.is_sub_brand', '0')
            ;
    }

    public function slugUrl() {
        return route('collection.slug', [$this->slug]);
    }

    public function getParentRelationName() {
        return null;
    }

    public function getNameAttribute($name) {
        if (isset($name) && strlen($name) > 0) {
            return $name;
        }

        $name = $this->brand->name;
        if (isset($name) && strlen($name) > 0) {
            return $name;
        }

        return $this->slug;
    }


    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait, ManyToManyRevisionableTrait;
}
