<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 1/19/2017
 * Time: 10:35 AM
 */

namespace Marcolin\Listeners;

use Illuminate\Support\Facades\Mail;
use Marcolin\Events\OrderWasPlaced;

/**
 * Sends an order confirmation email to the customer.
 *
 * Class EmailOrderConfirmation
 * @package Marcolin\Listeners
 */
class EmailOrderConfirmation
{
    public function construct()
    {

    }

    public function handle(OrderWasPlaced $event)
    {
        // Get the list of To addresses (customers).
        $toEmailList = [];
        if (config('app.debug') === false) {
            if (!$event->order->isGuestAccountOrder()) {
                // If the customer has an email address, add it.
                if (strlen($event->order->customer->email)) $toEmailList[strtoupper($event->order->customer->email)] = $event->order->customer->email;

                // If the customer has any users registered on the B2B site, use those email addresses too.
                foreach ($event->order->customer->webUsers as $webUser) {
                    if (strlen($webUser->email)) $toEmailList[strtoupper($webUser->email)] = $webUser->email;
                }
            }
        }

        // Finally, send to any addresses in the config/env file. Guest accounts go to a different email than non-guest accounts
        if ($event->order->isGuestAccountOrder()) {
            foreach (config('marcolin.emailDestinations.guestOrderConfirmation.to', []) as $to) {
                if (strlen($to)) $toEmailList[strtoupper($to)] = $to;
            }
        } else {
            foreach (config('marcolin.emailDestinations.orderConfirmation.to', []) as $to) {
                if (strlen($to)) $toEmailList[strtoupper($to)] = $to;
            }
        }


        // Next get the list of CC addresses (reps).
        $ccEmailList = [];
        if (config('app.debug') === false) {
            // Always use the rep the order was created by if there was one.
            $orderSalesRep = $event->order->salesRep;
            if ($orderSalesRep !== null) {
                $email = $orderSalesRep->salesGroupEmail;
                $ccEmailList[strtoupper($email)] = $email;
            }

            // Then get any reps that sell the brands included in this order. This is for B2B orders mostly.
            if (!$event->order->isGuestAccountOrder()) {
                $brandArray = [];

                foreach ($event->order->salesOrderItems()->with('unit.styleRelation.brand')->has('unit.styleRelation.brand')->get() as $salesOrderItem) {
                    $brandID = $salesOrderItem->unit->styleRelation->brand->id;
                    if (!in_array($brandID, $brandArray)) {
                        foreach ($event->order->customer->activeSalesRepsByBrand($brandID) as $salesRep) {
                            $email = $salesRep->salesGroupEmail;
                            if (!is_null($email) && strlen($email) > 0) {
                                $ccEmailList[strtoupper($email)] = $email;
                                break;
                            }
                        }
                        $brandArray[] = $brandID;
                    }
                }
            }
        }


        // Gets the list of BCC address (us and customer servicer mostly)
        if ($event->order->isGuestAccountOrder()) {
            foreach (config('marcolin.emailDestinations.guestOrderConfirmation.bcc', []) as $bcc) {
                if (strlen($bcc)) $bccEmailList[strtoupper($bcc)] = $bcc;
            }
        } else {
            foreach (config('marcolin.emailDestinations.orderConfirmation.bcc', []) as $bcc) {
                if (strlen($bcc)) $bccEmailList[strtoupper($bcc)] = $bcc;
            }
        }

        if (count($toEmailList) > 0 || count($bccEmailList) > 0) {
            Mail::send('marcolin::emails.orders.confirmation', ['order' => $event->order, 'url' => config('marcolin.b2b.url')], function ($email) use ($event, $toEmailList, $ccEmailList, $bccEmailList) {
                foreach ($toEmailList as $to) {
                    $email->to($to);
                }

                foreach ($ccEmailList as $cc) {
                    $email->cc($cc);
                }

                foreach ($bccEmailList as $bcc) {
                    $email->bcc($bcc);
                }

                $email->from(config('mail.from_orders.address'), config('mail.from_orders.name'));

                if ($event->order->orderType === 'B') {
                    $email->subject(trans('marcolin::email.orderConfirmationSubjectB2B', ['orderNumber' => $event->order->orderNumber, 'customerNumber' => $event->order->customer->customerCode]));
                } else {
                    $email->subject(trans('marcolin::email.orderConfirmationSubjectESB', ['orderNumber' => $event->order->orderNumber, 'customerNumber' => $event->order->customer->customerCode]));
                }
            });
        }
    }
}