<?php

namespace Marcolin\Models;

use Auth;
use DB;

class WebCart extends WebModel
{
    protected $table = 'WebCarts';
    protected $fillable = ['quantity', 'itemNote'];

    public function unit()
    {
        return $this->belongsTo(Unit::class, 'unit_id', 'id');
    }

    public function webUser()
    {
        return $this->belongsTo(WebUser::class);
    }

    public function scopeForUser($query, $webuserId)
    {
        return $query->where('webuser_id', $webuserId);
    }

    public function getExtendedPrice($customer)
    {
        return $this->unit->getCustomerPrice($customer) * $this->quantity;
    }

    public static function add(Unit $unit, $quantity = 1)
    {
        $instance = new static;
        $instance->webuser_id = Auth::id();
        $instance->unit_id = $unit->id;
        $instance->quantity = $quantity;
        $instance->save();
        return $instance;
    }

    public static function remove($userId, $unit_id)
    {
        $instance = new static;
        return $instance::where(['unit_id' => $unit_id, 'webuser_id' => $userId])->delete();
    }

    public static function cartItems($webUserId = false, $billToCustomerSegmentId = false)
    {
        // Get the web user. If and ID is provided, use it, otherwise, use who ever is logged in.
        if ($webUserId === false)
            $webUser = auth()->user();
        else
            $webUser = WebUser::find($webUserId);
        if (is_null($webUser)) return [];

        // Get the sold to customer.
        $soldToCustomer = b2b()->activeCustomer();

        $cartItems = self::forUser($webUser->id)
            ->whereHas("unit", function ($query) {
                return $query
                    ->webVisible();
            })
            ->with("unit")
            ->get();

        $triggeredPromotions = self::getTriggeredPromotionsInfo($soldToCustomer, $cartItems);

        foreach ($cartItems as $key => $cartItem) {
            $cartItems[$key]->customerPrice = $cartItem->unit->getCustomerPrice($soldToCustomer);
            // get customer's discount
            $discountPercent = $cartItem->unit->getCustomerDiscount($soldToCustomer);
            // get any promotional discounts
            foreach ($triggeredPromotions as $promotion) {
                $itemDiscountActions = $promotion->itemDiscountActions;
                if (isset($itemDiscountActions)) {
                    foreach ($itemDiscountActions as $itemDiscountAction) {
                        $discountPercent += $itemDiscountAction->discount_percent;
                    }
                }
                $orderDiscountActions = $promotion->orderDiscountActions;
                if (isset($orderDiscountActions)) {
                    foreach ($orderDiscountActions as $orderDiscountAction) {
                        $discountPercent += $orderDiscountAction->discount_percent;
                    }
                }
            }

            if ($discountPercent >= 100) $discountPercent = 100;

            if ($discountPercent > 0) {
                $cartItems[$key]->discountedPrice = round($cartItems[$key]->customerPrice - ($cartItems[$key]->customerPrice * ($discountPercent / 100)), 2);
            } else {
                $cartItems[$key]->discountedPrice = $cartItems[$key]->customerPrice;
            }

            $cartItems[$key]->calculatedPrice = $cartItems[$key]->discountedPrice * $cartItems[$key]->quantity;
        }

        return $cartItems;
    }

    public static function getCartDiscount()
    {
        return rulesChecker::checkPromoRules(app('generalPromos'));
    }

    public static function cartStatus(WebUser $webUser = null) {
        if (is_null($webUser)) $webUser = auth()->user();
        $soldToCustomer = b2b()->activeCustomer();

        $status = [
            'itemsCount' => 0,
            'quantityCount' => 0,
            'baseTotal' => 0,
            'discountedTotal' => 0,
            'discounts' => [
                'customer' => 0,
            ],
            'taxAmount' => 0,
            'shippingAmount' => 0,
            'shippingDiscountAmount' => 0,
            'shippingTaxAmount' => 0,
            'requireApproval' => false,
        ];

        $webCartItems = self::cartItems();

        $almostQualifyingPromotionsInfo = self::getAlmostQualifiedPromotionsInfo($soldToCustomer, $webCartItems);
        $status['almostQualifiedPromotionsInfo'] = $almostQualifyingPromotionsInfo;

        $triggeredPromotions = self::getTriggeredPromotionsInfo($soldToCustomer, $webCartItems);
        $status['triggeredPromotions'] = $triggeredPromotions;
        return $status;
    }

    public static function getAlmostQualifiedPromotionsInfo($customer, $webCartItems) {
        $promotions = Promotion::getActivePromotions();
        $almostQualifyingPromotionsInfo = [];
        $i = 0;
        foreach($promotions as $promotion) {
            $triggers = $promotion->triggers;
            $almostQualifiedValid = true; // This determines if there is a non-AQ trigger that is not satisfied.

            $j = 0;
            $promo = []; // Stores the Promotion and List of Triggers associated.
            $trigList = []; // Stores the Triggers w/ Qualifying Fields.
            foreach($triggers as $trigger) {

                if ($trigger->details->hasAlmostQualifyingField()) {
                    $trigDetails = [];
//                    For every promotion that has at least one trigger for an almost qualifying field, we create an object
//                      that stores the promotion, the triggerdetails, and the current cart status in terms of the relevant amounts.
                    $amountLeft = $trigger->details->isAlmostQualifying($customer, $webCartItems);
                    if ($amountLeft > 0) {
                        $trigDetails['trigger'] = $trigger; // Stores individual trigger and the amount progress their tied to.
                        if($trigger->details->minimum_quantity > 0 ) {
                            $trigDetails['type'] = 'minimum_quantity';
                        }else if($trigger->details->minimum_dollar > 0){
                            $trigDetails['type'] = 'minimum_dollar';
                        }
                        $trigDetails['amountLeft'] = $amountLeft;
                        $trigList[$j] = $trigDetails;        //Stores all Triggers that are tied to the individual promotion.
                        $j++;
                    }
                }else{
                    if(!$trigger->details->isTriggered($customer, $webCartItems)){
                        $almostQualifiedValid = false;
                    }
                }
            }

            if(count($trigList) and $almostQualifiedValid){
                $promo['triggers'] = $trigList;
                $promo['promotion'] = $promotion;
                $almostQualifyingPromotionsInfo[$i] = $promo;
                $i++;
            }
        }
        return $almostQualifyingPromotionsInfo;
    }

    public static function getTriggeredPromotionsInfo($customer, $webCartItems) {
        $promotions = Promotion::getActivePromotions();
        $triggeredPromotions = array();

        foreach($promotions as $promotion) {
            $triggeredAllTriggers = true;
            $triggers = $promotion->triggers;

            foreach($triggers as $trigger) {
                $triggeredAllTriggers = $triggeredAllTriggers && $trigger->details->isTriggered($customer, $webCartItems);
            }

            if($triggeredAllTriggers) {
                $triggeredPromotions[] = $promotion;
            }
        }

        return $triggeredPromotions;
    }

}
