<?php

namespace Marcolin\Models;

use Auth;
use DB;
use Image as Image2;
use Storage;

/**
 * Marcolin\Models\Product
 *
 * @property mixed $id
 */
class Customer extends WebModel
{
    protected $table = 'Customers';

    public function brands()
    {
        return $this->belongsToMany(Brand::class, 'Brand_Customer')->wherePivot('deleted_at', '=', '0000-00-00 00:00:00');
    }

    public function jobsonData()
    {
        return $this->hasOne(JobsonData::class, 'customer_id', 'id');
    }

    public function salesDistrict()
    {
        return $this->belongsTo(SalesDistrict::class, 'salesDistrict_id', 'id');
    }

    public function creditStatements()
    {
        return $this->hasMany(CreditStatement::class, 'customerCode_id', 'customerCode_id');
    }

    public function parentCustomer()
    {
        return $this->belongsTo(Customer::class, 'parent_customerCode_id', 'id');
    }

    public function childCustomers()
    {
        return $this->hasMany(Customer::class, 'parent_customerCode_id', 'parent_customerCode_id');
    }

    public function childCustomersShippable()
    {
        return $this->hasMany(Customer::class, 'parent_customerCode_id', 'parent_customerCode_id')
            ->where('shippable' ,'=', 1)
            ;
    }

    public function allSalesReps()
    {
        $salesReps = SalesRep::where('active', '=', 1)
            ->whereHas('salesDistrict.customers', function ($q) {
                return $q->where('id', '=', $this->id);
            })
            ->with('brands');
        return $salesReps->get();
    }

    public function activeSalesReps()
    {
        $brands = $this->brands->pluck('id');

        $salesReps = SalesRep::where('active', '=', 1)
            ->whereHas('salesDistrict.customers', function ($q) use ($brands) {
                return $q
                    ->where('Customers.id', '=', $this->id)
                    ->whereIn('SalesRep_Customer_Brand.brand_id', $brands);
            })
            ->with('brands');

        return $salesReps->get();
    }

    public function activeSalesRepsByBrand($brand_id)
    {
        $salesReps = SalesRep::where('active', '=', 1)
            ->where('deleted_at', '=', '0000-00-00 00:00:00')
            ->whereHas('salesDistrictTrashed.customersTrashed', function ($q) use ($brand_id) {
                return $q
                    ->where('Customers.id', '=', $this->id)
                    ->where('SalesRep_Customer_Brand.brand_id', '=', $brand_id);
            })
            ->with('brands');

        return $salesReps->get();
    }

    public function customerDiscounts()
    {
        return $this->hasMany(CustomerDiscount::class, 'customer_id', 'id');
    }

    public function customerLocation() {
        return $this->hasOne(CustomerLocation::class, 'customer_id', 'id');
    }

    public function orders()
    {
        return $this->hasMany(Order::class, 'soldTo_customer_id', 'id');
    }

    public function salesOrders()
    {
        return $this->hasMany(SalesOrder::class, 'customer_id', 'id');
    }

    public function payerCustomer()
    {
        return $this->hasOne(Customer::class, 'customerCode', 'payerCode');
    }

    public function webUsers()
    {
        return $this->hasMany(WebUser::class, 'customer_id', 'id');
    }

    public function lastOrderedStyle($style)
    {
        return $this->orderHistories->where('style', $style)->sortByDesc('billingDate')->first();
    }

    public function getFormattedTelephoneAttribute()
    {
        return preg_replace('~.*(\d{3})[^\d]{0,7}(\d{3})[^\d]{0,7}(\d{4}).*~', '($1) $2-$3', $this->telephone) . "\n";
    }

    public function getFormattedTelephone2Attribute()
    {
        return preg_replace('~.*(\d{3})[^\d]{0,7}(\d{3})[^\d]{0,7}(\d{4}).*~', '($1) $2-$3', $this->telephone2) . "\n";
    }

    public function getFormattedFaxAttribute()
    {
        return preg_replace('~.*(\d{3})[^\d]{0,7}(\d{3})[^\d]{0,7}(\d{4}).*~', '($1) $2-$3', $this->telephone2) . "\n";
    }

    public function getProductVisibilityAttribute()
    {
        if ($this->brands->where('brandCode', 'HM')->count() > 0) {
            return 'harley';
        } else {
            return 'standard';
        }
    }

    public function getLockedBrands()
    {
        $brands = $this->brands->pluck('id');
        $lockedBrands = Brand::webVisible()->whereNotIn('id', $brands)->where('deleted_at', '=', '0000-00-00')->orderBy('brandName')->get();
        return $lockedBrands;
    }

    public function recentOrders($count = 5)
    {
        return $this
            ->orders()
//            ->submitted()
            ->with('orderItems')
            ->orderByDate()
            ->take($count)
            ->get();
    }

    public function topPurchasedItems($count = 10)
    {
        $orders = $this
            ->orders()
            ->has('orderItems.unit.styleRelation.webStyle.webVisibleWebUnits.assets')
            ->with([
                'orderItems' => function ($q) {
                    return $q
                        ->has('unit.styleRelation.webStyle.webVisibleWebUnits.assets')
                        ->with([
                            'unit.styleRelation.webStyle.style',
                            'unit.styleRelation.webStyle.webVisibleWebUnits.assets',
                            'unit.styleRelation.webStyle.webVisibleWebUnits.unit.priceListValues',
                            'unit.styleRelation.webStyle.webVisibleWebUnits.unit.styleRelation',
                        ])
                        ;
                },
            ])
            ->orderBy('orderDate', 'desc')
            ->limit($count);

        $styles = [];
        foreach ($orders->get() as $order) {
            foreach ($order->orderItems as $orderItem) {
                $styles[] = $orderItem->unit->styleRelation->webStyle;
                if (count($styles) >= $count) {
                    break;
                }
            }

            if (count($styles) >= $count) {
                break;
            }
        }

        return $styles;
    }

    public function isVisionSource() {
        return $this->customerCodeHierarchy === '500142';
    }
}

