<?php

namespace Marcolin\Models;

use DB;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * Class PromotionCustomerTrigger
 * @package Marcolin\Models
 * @property boolean $target_all_customers
 * @property boolean $target_by_microsites
 */
class PromotionCustomerTrigger extends PromotionTriggerDetails {
    protected $table = 'PromotionCustomerTriggers';
    protected $fillable = ['target_all_customers', 'target_by_microsites', 'commit_sequence'];
    protected $casts = [
        'target_all_customers' => 'boolean',
        'target_by_microsites' => 'boolean',
    ];

    // Overrides Model::boot(), used to define a specific 'deleted' state
    public static function boot() {
        parent::boot();
        // make sure all related rows in PromotionCustomerTrigger_Customer and PromotionCustomerTrigger_WebSilo are also deleted
        static::deleted(function (PromotionCustomerTrigger $promotionCustomerTrigger) {
            $promotionCustomerTriggerCustomers =
                PromotionCustomerTrigger_Customer
                    ::where('promotioncustomertrigger_id', '=', $promotionCustomerTrigger->id)
                    ->get()
            ;

            foreach ($promotionCustomerTriggerCustomers as $promotionCustomerTriggerCustomer) {
                $promotionCustomerTriggerCustomer->delete();
            }

            $promotionCustomerTriggerWebSilos =
                PromotionCustomerTrigger_WebSilo
                    ::where('promotioncustomertrigger_id', '=', $promotionCustomerTrigger->id)
                    ->get()
            ;

            foreach ($promotionCustomerTriggerWebSilos as $promotionCustomerTriggerWebSilo) {
                $promotionCustomerTriggerWebSilo->delete();
            }
        });
    }

    /**
     * Get relationship to list of customers defined in PromotionCustomerTrigger_Customer
     * @return BelongsToMany
     */
    public function customers() {
        return $this->belongsToMany(
            Customer::class, 'PromotionCustomerTrigger_Customer',
            'promotioncustomertrigger_id', 'customer_id')
            ->wherePivot('deleted_at', '=', DB::raw(0))
            ->withPivot([
                'excluded',
            ])
            ;
    }

    /**
     * Get relationship to list of microsites defined in PromotionCustomerTrigger_WebSilo
     * @return BelongsToMany
     */
    public function webSilos() {
        return $this->belongsToMany(
            WebSilo::class, 'PromotionCustomerTrigger_WebSilo',
            'promotioncustomertrigger_id', 'websilo_id')
            ->wherePivot('deleted_at', '=', DB::raw(0))
            ;
    }

    /**
     * @return HasMany
     */
    public function promotionCustomerTriggerWebSilos() {
        return $this->hasMany(PromotionCustomerTrigger_WebSilo::class, 'promotioncustomertrigger_id', 'id');
    }

    /**
     * Triggered logic for the customer trigger:
     * - return true if targeting all customers
     * or if the sold-to customer is in the targeted list
     * or if the sold-to customer belongs to one of the targeted microsites
     * @param Customer $soldToCustomer
     * @param array $webCartItems
     * @return bool
     */
    public function isTriggered(Customer $soldToCustomer, $webCartItems = []) {
        $isHarley = $soldToCustomer->product_visibility == 'harley';

        if ($this->target_by_microsites) {
            if ($isHarley) {
                $harleyTarget = $this->promotionCustomerTriggerWebSilos()->where('websilo_id', WebSilo::getHarleyDavidsonSite()['id'])->first();
                return isset($harleyTarget);
            }

            $customersWebSilo = $soldToCustomer->webSiloWithoutLandingPage()->first();
            if (!isset($customersWebSilo)) {
                $buyingGroup = $soldToCustomer->buyingGroup;
                if (isset($buyingGroup)) {
                    $buyingGroupWebSilos = $buyingGroup->webSilo;
                    if ($buyingGroupWebSilos && count($buyingGroupWebSilos) > 0) {
                        $customersWebSilo = $buyingGroupWebSilos[0];
                    }
                }

                if (!isset($customersWebSilo)) {
                    $customersWebSilo = WebSilo::defaultWebSilo()->first(); // use defaultWebSilo() scope
                }
            }
            $webSilo =
                $this->webSilos()
                    ->wherePivot('websilo_id', '=', $customersWebSilo->id)
                    ->first()
            ;
            return !is_null($webSilo);
        } else if ($isHarley) {
            return false;
        } else if ($this->target_all_customers) {
            return true;
        } else {
            $customer =
                $this->customers()
                    ->wherePivot('customer_id', '=', $soldToCustomer->id)
                    ->first()
            ;

            return !is_null($customer);
        }
    }

    /**
     * Update the relevant PromotionCustomerTrigger_Customer row to 'redeemed'
     * @param Customer $soldToCustomer
     * @return mixed|void
     */
    public function postTriggerOperation(Customer $soldToCustomer) {}

    /**
     * Boolean, any promo trigger that can use the Almost Qualifying Header should implement this and return 'true'
     * otherwise return 'false'
     * @return bool
     */
    public function hasAlmostQualifyingField() {
        return false;
    }

    /**
     * Any promo trigger that can use the Almost Qualifying Header should implement this and
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return bool
     */
    public function isAlmostQualifying(Customer $soldToCustomer, $webCartItems) {
        return false;
    }
}
