<?php

namespace Marcolin\Models;

use Illuminate\Database\Eloquent\Builder;

/**
 * Class Promotion
 * @package Hilco\Models
 *
 * @method static Builder active()
 */
class PromotionCurrentOrderQuantityTrigger extends PromotionTriggerDetails {
    protected $table = 'PromotionCurrentOrderQuantityTriggers';
    protected $fillable = ['minimum_quantity', 'any_product', 'almost_qualified', 'include_backorders', 'commit_sequence'];

    // Overrides Model::boot(), used to define a specific 'deleted' state
    public static function boot() {
        parent::boot();
        // make sure all related rows in PromotionCurrentOrderQuantityTrigger_Product are also deleted
        static::deleted(function (PromotionCurrentOrderQuantityTrigger $promotionCurrentOrderQuantityTrigger) {
            $promotionCurrentOrderQuantityTriggerProduct =
                PromotionCurrentOrderQuantityTrigger_Product
                    ::where('promotioncurrentorderquantitytrigger_id', '=', $promotionCurrentOrderQuantityTrigger->id)
                    ->get();

            foreach ($promotionCurrentOrderQuantityTriggerProduct as $promotionCurrentOrderValueTriggerProduct) {
                $promotionCurrentOrderValueTriggerProduct->delete();
            }
        });
    }

    /**
     * Boolean, all promo trigger types need to implement to use their specific logic
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return mixed
     */
    public function isTriggered(Customer $soldToCustomer, $webCartItems = []) {
        $totalItemQuantity = $this->getCurrentQualifyingCartTotal($webCartItems);

        return $totalItemQuantity >= $this->minimum_quantity;
    }

    /**
     * promo trigger types should implement if there post operations that should happen after promo is applied
     * @param $soldToCustomer
     * @return mixed
     */
    public function postTriggerOperation(Customer $soldToCustomer) {
        // TODO: Implement postTriggerOperation() method.
    }

    /**
     * Boolean, any promo trigger that can use the Almost Qualifying Header should implement this and return 'true'
     * otherwise return 'false'
     * @return bool
     */
    public function hasAlmostQualifyingField() {
        return true;
    }

    /**
     * Any promo trigger that can use the Almost Qualifying Header should implement this and
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return mixed
     */
    public function isAlmostQualifying(Customer $soldToCustomer, $webCartItems) {
        $totalItemQuantity = $this->getCurrentQualifyingCartTotal($webCartItems);

        return $totalItemQuantity < $this->minimum_quantity; //&& $totalItemQuantity >= $this->almost_qualified;
    }

    /**
     * Get all WebBrands specifically targeted by this trigger
     * @return mixed
     */
    public function webBrands() {
        return $this
            ->morphedByMany(WebBrand::class, 'product',
                'PromotionCurrentOrderQuantityTrigger_Product', 'promotioncurrentorderquantitytrigger_id')
            ->where('PromotionCurrentOrderQuantityTrigger_Product.deleted_at', '=', '0000-00-00 00:00:00')
            ;
    }

    /**
     * Get all WebStyles specifically targeted by this trigger
     * @return mixed
     */
    public function webStyles() {
        return $this
            ->morphedByMany(WebStyle::class, 'product',
                'PromotionCurrentOrderQuantityTrigger_Product', 'promotioncurrentorderquantitytrigger_id')
            ->where('PromotionCurrentOrderQuantityTrigger_Product.deleted_at', '=', '0000-00-00 00:00:00')
            ;
    }

    /**
     * Get all WebUnits specifically targeted by this trigger
     * @return mixed
     */
    public function webUnits() {
        return $this
            ->morphedByMany(WebUnit::class, 'product',
                'PromotionCurrentOrderQuantityTrigger_Product', 'promotioncurrentorderquantitytrigger_id')
            ->where('PromotionCurrentOrderQuantityTrigger_Product.deleted_at', '=', '0000-00-00 00:00:00')
            ;
    }

    /**
     * Gets the total number of 'qualifying' items in the cart, where 'qualifying' is defined as
     * - contained in the list of targeted web products (units, styles, brands), if not targeting all products
     * @param $webCartItems
     * @return int
     */
    public function getCurrentQualifyingCartTotal($webCartItems) {
        $totalItemQuantity = 0;

        if (isset($webCartItems) && count($webCartItems) > 0) {
            if ($this->any_product) {
                foreach ($webCartItems as $webCartItem) {
                    if ($this->include_backorders || !$webCartItem->cartable->isBackOrdered()) {
                        $totalItemQuantity = $totalItemQuantity + $webCartItem->quantity;
                    }
                }
            } else {
                $relevantUnits = $this->webUnits;
                $relevantStyles = $this->webStyles;
                $relevantBrands = $this->webBrands;

                foreach ($webCartItems as $webCartItem) {
                    if ($webCartItem->cartable_type == 'unit') {
                        $webUnit = $webCartItem->cartable;
                        $webStyle = $webUnit->webStyle;
                        $webBrand = $webStyle->webBrand;
                        if ($this->include_backorders || !$webCartItem->cartable->isBackOrdered()) {
                            if ((isset($webUnit) && $relevantUnits->contains($webUnit)) ||
                                (isset($webStyle) && $relevantStyles->contains($webStyle)) ||
                                (isset($webBrand) && $relevantBrands->contains($webBrand))
                            ) {
                                $totalItemQuantity = $totalItemQuantity + $webCartItem->quantity;
                            }
                        }
                    }
                }
            }
        }

        return $totalItemQuantity;
    }
}
