<?php

namespace Marcolin\Models;

use Auth;
use Barryvdh\Debugbar\DataCollector\LaravelCollector;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Log;

class WebCart extends WebModel
{
    protected $table = 'WebCarts';
    protected $fillable = ['quantity', 'itemNote', 'shippingMethod', 'customeraddress_id', 'add_item_source', 'saved_for_later'];
    protected $casts = [
        'saved_for_later' => 'boolean',
    ];

    protected static $morphMap = [
        /* 'unit' => Unit::class --- this morph mapping already exists in WebAssets, so it's not defined here */
        'prepack' => Prepack::class
    ];

    const ITEM_SOURCE_PREPACK           = 'prepack';
    const ITEM_SOURCE_ADVANCED_SEARCH   = 'advanced search';
    const ITEM_SOURCE_QUICK_ORDER       = 'quick order';
    const ITEM_SOURCE_REORDER           = 'reorder';
    const ITEM_SOURCE_HOME_PAGE         = 'home page';
    const ITEM_SOURCE_PRODUCT_DETAILS   = 'product details';
    const ITEM_SOURCE_ORDER_HISTORY     = 'order history';
    const ITEM_SOURCE_FAVORITES         = 'favorites';

    public function cartable() {
        return $this->morphTo();
    }

    /**
     * Allows a whereHas() type of relationship call to be made on this polymorphic model (see WebCart::unitCartItems)
     * @param $query
     * @param $type
     * @param $callback
     */
    public function scopeWhereCartable ($query, $type, $callback) {
        return $this->filterPolymorphicRelation($query, 'cartable', $type, $callback);
    }

    /**
     * To be used in conjunction with the whereCartable scope to implement a version of whereHas() that can be used
     * on WebCarts since it's a polymorphic model (see WebCart::unitCartItems)
     * @param $query
     * @param $relationName
     * @param $type
     * @param $callback
     */
    protected function filterPolymorphicRelation ($query, $relationName, $type, $callback) {
        $class = $this->getActualClassNameForMorph($type);

        $query->where("{$relationName}_type", '=', $type)
            ->whereIn("{$relationName}_id", function ($query) use ($class, $callback) {
                $model = new $class;
                $builder = new Builder($query);

                $query = $builder->setModel($model);

                $callback($query);

                $query->select([$model->getKeyName()]);
            });
    }

    public static function getMorphMap()
    {
        return self::$morphMap;
    }

    public function webUser()
    {
        return $this->belongsTo(WebUser::class);
    }

    public function customerAddress()
    {
        return $this->belongsTo(CustomerAddress::class, 'customeraddress_id', 'id');
    }

    public function scopeForUser($query, $webuserId)
    {
        return $query->where('webuser_id', $webuserId);
    }

    public function scopeForUserByCartableType($query, $webuser_id, $cartable_type){
        return $query->forUser($webuser_id)->where('cartable_type', $cartable_type);
    }

    public function scopeSavedForUser($query, $webuserId) {
        return $query->forUser($webuserId)->where('saved_for_later', 1);
    }

    public function scopeActiveForUser($query, $webuserId) {
        return $query->forUser($webuserId)->where('saved_for_later', 0);
    }

    public function getExtendedPrice() {
        return $this->cartable->getCustomerPrice() * $this->quantity;
    }

    public static function add($cartable, $quantity = 1, $trayinfo = null, $itemSource = null) {
        $instance = new static;

        $instance->webuser_id       = Auth::id();
        $instance->cartable_type    = $cartable->getMorphClass();
        $instance->cartable_id      = $cartable->id;
        $instance->quantity         = $quantity;
        $instance->itemNote         = $trayinfo;
        $instance->add_item_source  = $itemSource;

        $soldToCustomer = b2b()->activeCustomer();
        $instance->shippingMethod = $soldToCustomer->preferredShippingMethodCode;

        if($soldToCustomer->defaultShippingAddress != null){
            $instance->customerAddress_id = $soldToCustomer->defaultShippingAddress->id;
        }else{
            $instance->customerAddress_id = null;
            Log::error("Customer doesn't have a default shipping address. ID-".$soldToCustomer->id);
        }


        $instance->save();
        return $instance;
    }

    public static function remove($userId, $cartable_id, $cartable_type)
    {
        $instance = new static;
        return $instance::where(['cartable_id' => $cartable_id, 'cartable' => $cartable_type, 'webuser_id' => $userId])->delete();
    }

    public static function restoreSaved ($webuserId) {
        $instance = new static;
        return $instance::forUser($webuserId)->where('saved_for_later', 1)->update(['saved_for_later' => false]);
    }

    public static function cartItems($webUserId = false, $billToCustomerSegmentId = false) {
        // Get the web user. If an ID is provided, use it, otherwise, use who ever is logged in.
        if ($webUserId === false)
            $webUser = auth()->user();
        else
            $webUser = WebUser::find($webUserId);

        if (is_null($webUser)) return [];

        // Get the sold to customer.
        $soldToCustomer = b2b()->activeCustomer();

        $cartUnits = self::activeForUser($webUser->id)
            ->whereCartable(Unit::getModel()->getMorphClass(), function ($query) {
                return $query->webVisible();
            })
            ->with([
                'cartable' => function ($query) {
                    return $query
                        ->with([
                            'webUnit' => function ($query) {
                                return $query
                                    ->with([
                                        'assets',
                                        'webStyle',
                                    ]);
                            },
                            'webStyle' => function ($query) {
                                return $query
                                    ->with([
                                        'style',
                                    ]);
                            }
                        ]);
                },
            ])
            ->get()
        ;

        $cartPrepacks = self::activeForUser($webUser->id)
            ->whereCartable(Prepack::getModel()->getMorphClass(), function ($query) {
                return $query->isActive();
            })->with(['cartable' => function ($query) {
                return $query->with(['prepackItems' => function ($query) {
                    return $query->with(['unit' => function ($query) {
                        return $query->with(['webUnit.assets']);
                    }]);
                }]);
            }])->get()
        ;



        $cartItems = $cartUnits->merge($cartPrepacks);

        $priceOverrideAction = false;
//        2021-09-02: Nick told me to get rid of this in order to fix a bug
//        if (b2b()->activePromotion() && b2b()->activePromotion()->isQualifying($soldToCustomer, $cartItems)) {
//            $priceOverrideAction = b2b()->activePromotion()->priceOverrideActions()->first();
//        }

        foreach ($cartItems as $key => $cartItem) {
            $cartItems[$key]->customerPrice = $cartItem->cartable->getCustomerPrice($soldToCustomer);
            $cartItems[$key]->discountPercent = $cartItem->cartable->getCustomerDiscount($soldToCustomer);

            if ($priceOverrideAction && $priceOverrideAction->isApplicable($cartItem->cartable)) {
                $cartItems[$key]->discountedPrice = $priceOverrideAction->override_amount;
            } else if ($cartItems[$key]->discountPercent > 0) {
                $cartItems[$key]->discountedPrice = round($cartItems[$key]->customerPrice - ($cartItems[$key]->customerPrice * ($cartItems[$key]->discountPercent / 100)), 2);
            } else {
                $cartItems[$key]->discountedPrice = $cartItems[$key]->customerPrice;
            }

            $cartItems[$key]->calculatedPrice = $cartItems[$key]->discountedPrice * $cartItems[$key]->quantity;
        }
        return $cartItems;
    }

    public static function getCartDiscount()
    {
        return rulesChecker::checkPromoRules(app('generalPromos'));
    }

}
