<?php

namespace Marcolin\Models;

use Illuminate\Support\Facades\DB;

/**
 * Class PromotionCustomerTrigger
 * @package Marcolin\Models
 */
class PromotionCustomerTrigger extends PromotionTriggerDetails {
    protected $table = 'PromotionCustomerTriggers';
    protected $fillable = ['target_all_customers', 'commit_sequence'];

    // Overrides Model::boot(), used to define a specific 'deleted' state
    public static function boot() {
        parent::boot();
        // make sure all related rows in PromotionCustomerTrigger_Customer are also deleted
        static::deleted(function (PromotionCustomerTrigger $promotionCustomerTrigger) {
            $promotionCustomerTriggerCustomers =
                PromotionCustomerTrigger_Customer
                    ::where('promotioncustomertrigger_id', '=', $promotionCustomerTrigger->id)
                    ->get()
            ;

            foreach ($promotionCustomerTriggerCustomers as $promotionCustomerTriggerCustomer) {
                $promotionCustomerTriggerCustomer->delete();
            }
        });
    }

    /**
     * Get relationship to list of customers defined in PromotionCustomerTrigger_Customer
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function customers() {
        return $this->belongsToMany(
            Customer::class, 'PromotionCustomerTrigger_Customer',
            'promotioncustomertrigger_id', 'customer_id')
            ->wherePivot('deleted_at', '=', DB::raw(0))
            ->withPivot([
                'excluded',
            ])
            ;
    }

    /**
     * Triggered logic for the customer trigger:
     * - return true if targeting all customers or if the sold-to customer is in the targeted list
     * @param Customer $soldToCustomer
     * @param array $webCartItems
     * @return bool|mixed
     */
    public function isTriggered(Customer $soldToCustomer, $webCartItems = []) {
        if ($this->target_all_customers) {
            return true;
        } else {
            $customer =
                $this->customers()
                    ->wherePivot('customer_id', '=', $soldToCustomer->id)
                    ->first()
            ;

            return !is_null($customer);
        }
    }

    /**
     * Update the relevant PromotionCustomerTrigger_Customer row to 'redeemed'
     * @param Customer $soldToCustomer
     * @return mixed|void
     */
    public function postTriggerOperation(Customer $soldToCustomer) {}

    /**
     * Boolean, any promo trigger that can use the Almost Qualifying Header should implement this and return 'true'
     * otherwise return 'false'
     * @return bool
     */
    public function hasAlmostQualifyingField() {
        return false;
    }

    /**
     * Any promo trigger that can use the Almost Qualifying Header should implement this and
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return bool
     */
    public function isAlmostQualifying(Customer $soldToCustomer, $webCartItems) {
        return false;
    }
}
