<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 1/27/16
 * Time: 4:40 PM
 */

namespace Visionware\DataManager;

class SchemaCreator {
    protected $definition;
    protected $drops;
    protected $use_default_statement = false;

    public function __construct($definition, $drops = false) {
        $this->definition = $definition;
        $this->drops = $drops;
        $this->table_name_postfix = '';
    }

    public function build_schema() {
        $output = [];
        foreach ($this->definition['tables'] as $table) {
            $table = $this->process_table($table);
            if ($table === false) continue;

            $columns = $this->build_columns($table);
            $indices = $this->build_indices($table);

            $output = array_merge($output, $this->build_statements($table['name'] . $this->table_name_postfix, $columns, $indices));
        }

        return $output;
    }

    protected function process_table($table) {
        return $table;
    }

    protected function build_columns($table) {
        $columns = [];
        foreach ($table['columns'] as $column) {
            $column = $this->process_column($column);
            if ($column === false) continue;

            $column_string = "`$column[name]` $column[type]";

            if (!$column['null']) $column_string .= ' NOT NULL';


            if (!is_null($column['default'])) $column_string .= " DEFAULT $column[default]";
            else if ($column['null']) $column_string .= " DEFAULT NULL";

            if ($column['extra']) $column_string .= " $column[extra]";

            $columns[] = $column_string;
        }

        return $columns;
    }

    protected function process_column($column) {
        return $column;
    }

    protected function build_indices($table) {
        $indices = [];
        foreach ($table['indices'] as $index) {
            $index = $this->process_index($index);
            if ($index === false) continue;
            $index_string = '';

            if ($index['name'] == 'PRIMARY') {
                $index_string = 'PRIMARY ';
                $index['name'] = '';
            } else {
                $index['name'] = " `$index[name]`";
                if ($index['unique']) $index_string .= 'UNIQUE ';
            }
            uasort(
                $index['columns'],
                function ($a, $b) {
                    return $a['sequence'] - $b['sequence'];
                }
            );
            $columns_string = implode("`, `", array_pluck($index['columns'], 'name'));

            $index_string .= "KEY$index[name] (`$columns_string`)";
            $indices[] = $index_string;
        }

        return $indices;
    }

    protected function process_index($index) {
        return $index;
    }

    protected function build_statements($table, $columns, $indices) {
        $columns_string = implode(",\n  ", $columns);
        $indices_string = implode(",\n  ", $indices);

        $statements = [];
        $methods = get_class_methods($this);
        foreach ($methods as $method) {
            if (substr($method, 0, 10) == 'statement_') {
                $statements = array_merge($statements, call_user_func_array([$this, $method], [$table, $columns_string, $indices_string, $this->drops]));
            }
        }
        if (count($statements) == 0 || $this->use_default_statement) $statements = $this->default_statement_create($table, $columns_string, $indices_string, $this->drops);
        return $statements;
    }

    protected function default_statement_create($table, $columns_string, $indices_string, $drops) {
        $sql = [];
        if ($drops) $sql[] = "DROP TABLE IF EXISTS `$table`;";

        $sql[] = <<<SQL
CREATE TABLE `$table` (
  $columns_string,
  $indices_string
) ENGINE=InnoDB DEFAULT CHARSET=utf8;
SQL;
        return $sql;
    }
}