<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 4/18/16
 * Time: 11:09 AM
 */

namespace Visionware\DataManager\Schema;

use Exception;

class Schema implements SchemaInfoInterface {
    private $definition;

    public function __construct($definition) {
        $this->definition = $definition;
    }

    /**
     * Returns the entire definition array.
     * @return mixed
     */
    public function definition() {
        return $this->definition;
    }

    /**
     * Is called for each table when generating a list of tables. Override and return false to omit.
     * @param $table_name
     * @param $table
     * @return bool
     */
    protected function filter_table($table_name, $table) {
        return true;
    }

    /**
     * Is called for each field when generating a list of fields. Override and return false to omit fields.
     * @param $table_name
     * @param $table
     * @param $field_name
     * @param $field
     * @return bool
     */
    protected function filter_field($table_name, $table, $field_name, $field) {
        return true;
    }

    /**
     * Returns an array of table information, keyed on table name.
     * @return array
     */
    public function tables() {
        $tables = [];
        foreach ($this->definition['tables'] as $table) {
            if (!$this->filter_table($table['table'], $table)) continue;
            $tables[$table['table']] = $table;
        }
        return $tables;
    }

    /**
     * Returns a list of table names.
     * @return array
     */
    public function table_names() {
        $names = [];
        foreach ($this->tables() as $table_name => $table) {
            $names[] = $table_name;
        }
        return $names;
    }

    /**
     * Returns an array of table information.
     * @param $table_name
     * @return array
     */
    public function table($table_name) {
        return array_get($this->definition['tables'], $table_name);
    }

    /**
     * Returns an list of field information arrays, or just a list of field names if with_info is false.
     * @param $table_name
     * @param bool $with_info
     * @return array
     */
    public function fields($table_name, $with_info = true) {
        $table = $this->table($table_name);


        $fields = [];
        $key = $this->table_key($table_name);
        if (isset($table) && isset($table['fields'])) {
            foreach ($table['fields'] as $field) {
                if (!$this->filter_field($table['table'], $table, $field['name'], $field)) continue;
                if (in_array($field['name'], $key)) $field['nullable'] = false;
                if (!array_key_exists ('cst_in_version', $field)) $field['cst_in_version'] = 'all';

                if ($with_info) $fields[$field['name']] = $field;
                else $fields[] = $field['name'];
            }
        }
        return $fields;
    }

    /**
     * Returns an array of field information if it exists on the table.
     * @param $table_name
     * @param $field_name
     * @return bool|mixed
     */
    public function field($table_name, $field_name) {
        $fields = $this->fields($table_name);
        foreach ($fields as $field) {
            if ($field['name'] == $field_name) return $field;
        }
        return false;
    }

    /**
     * Returns a list of column information arrays.
     * @param $table_name
     * @return mixed
     */
    public function columns($table_name) {
        $columns = $this->prepend_columns($table_name);
        foreach ($this->fields($table_name) as $field_name => $field) {
            $columns[] = $this->column_from_field($field);
        }
        return array_merge($columns, $this->append_columns($table_name));
    }

    public function field_from_column($table_name, $column) {
        $column_name = array_get($column, 'name', true);
        foreach ($this->fields($table_name) as $field) {
            if (array_get($field, 'name', false) == $column_name) {
                return $field;
            }
        }
        return [];
    }

    /**
     * Returns a column information array.
     * @param $table_name
     * @param $column_name
     * @return array|mixed
     */
    public function column($table_name, $column_name) {
        foreach ($this->fields($table_name) as $field_name => $field) {
            if ($field['name'] == $column_name) return $this->column_from_field($field);
        }
        /* if it isn't an actual field, continue */
        foreach ($this->prepend_columns($table_name) as $column) {
            if ($column_name == $column['name']) return $column;
        }
        foreach ($this->append_columns($table_name) as $column) {
            if ($column_name == $column['name']) return $column;
        }
        die("Cannot locate column $column_name in table $table_name");
    }

    /**
     * Returns a list of column names.
     * @param $table_name
     * @return array
     */
    public function column_names($table_name) {
        $names = [];
        foreach ($this->columns($table_name) as $field) {
            $names[] = $field['name'];
        }
        return $names;
    }

    /**
     * Returns a list of index names.
     * @param $table_name
     * @return array
     */
    public function index_names($table_name) {
        $names = [];
        foreach ($this->indices($table_name) as $field) $names[] = $this->index_name($field);
        return $names;
    }

    /**
     * Returns the name of an index for the given index information array.
     * @param $index
     * @return mixed
     */
    public function index_name($index) {
        if (array_key_exists('name', $index) && strlen($index['name'])) return $index['name'];
        return implode('-', $index['columns']);
    }

    /**
     * Returns an index information array.
     * @param $table_name
     * @param $index_name
     * @return mixed
     */
    public function index($table_name, $index_name) {
        foreach ($this->indices($table_name) as $index) {
            if ($this->index_name($index) == $index_name) return $index;
        }
        die("Cannot locate index $index_name in table $table_name");
    }

    /**
     * Returns a list of index information arrays.
     * @param $table_name
     * @return mixed
     */
    public function indices($table_name) {
        $indices = $this->prepend_indices($table_name);
        if (count($this->table_key($table_name))) {
            $indices[] = $this->new_index('UNIQUE KEY', $this->table_key($table_name), 'unique_key');
        }
        foreach ($this->index_fields($table_name) as $index_name => $index) {
            $indices[] = $this->index_from_field($index);
        }
        foreach ($this->fields($table_name) as $field_name => $field) {
            if (array_key_exists('foreign', $field)) {
                $indices[] = $this->new_index('KEY', [$field_name], $field_name);
            }
        }
        return array_merge($indices, $this->append_indices($table_name));
    }

    /**
     * Returns the key for a table.
     * @param $table_name
     * @return array
     */
    public function table_key($table_name) {
        return array_key_exists('key', ($this->table($table_name) !== null ? $this->table($table_name) : [])) ? $this->table($table_name)['key'] : [];
    }

    /**
     * Returns a list of extra indices for an index.
     * @param $table_name
     * @param bool $with_info
     * @return array
     */
    public function index_fields($table_name, $with_info = true) {
        $table = $this->table($table_name);
        $indices = [];
        if (array_key_exists('indices', $table)) {
            foreach ($table['indices'] as $index) {
                if ($with_info) $indices[$this->index_name($index)] = $index;
                else $indices[] = $this->index_name($index);
            }
        }
        return $indices;
    }

    public function pk_type($table_name) {
        $table = $this->table($table_name);
        
        return array_get($table, 'pk_type', 'int');
    }
    
    /**
     * @param $table_name
     * @return bool
     */
    public function has_uuid($table_name) {
        return $this->pk_type($table_name) == 'uuid';
    }

    /**
     * @param $table_name
     * @return bool
     */
    public function has_bigint_id($table_name) {
        return $this->pk_type($table_name) == 'bigint';
    }

    public function has_int_id($table_name) {
        return $this->pk_type($table_name) == 'int';
    }

    /**
     * @param $table_name
     * @return bool
     */
    public function has_timestamps($table_name) {
        return !(array_key_exists('no_timestamps', $this->table($table_name)) && $this->table($table_name)['no_timestamps']);
    }

    /**
     * @param $table_name
     * @return bool
     */
    public function is_synced_up($table_name) {
        return (array_key_exists('is_synced_up', $this->table($table_name)) && $this->table($table_name)['is_synced_up']);
    }

    /**
     * @param $table_name
     * @return bool
     */
    public function is_synced_down($table_name) {
        return (array_key_exists('is_synced_down', $this->table($table_name)) && $this->table($table_name)['is_synced_down']);
    }

    /**
     * @param $table_name
     * @return bool
     */
    public function has_deleted($table_name) {
        return !(array_key_exists('no_deleted', $this->table($table_name)) && $this->table($table_name)['no_deleted']);
    }

    /**
     * Returns a column information array from the provided field information array.
     * @param $field
     * @return array
     */
    protected function column_from_field($field) {
        if (array_key_exists('foreign', $field)) {
            $type = $this->column($field['on_table'], $field['foreign'])['type'];
        } else {
            $type = $field['type'];
        }
        $nullable = array_key_exists('nullable', $field) ? $field['nullable'] : false;
        return [
            'name' => $field['name'],
            'type' => strtolower($type),
            'nullable' => $nullable,
            'default' => strtoupper(array_key_exists('default', $field) && strlen($field['default']) ? $field['default'] : ($nullable ? 'NULL' : '')),
            'extra' => strtoupper(array_key_exists('extra', $field) ? $field['extra'] : ''),
        ];
    }

    /**
     * Creates a column information array from parameters.
     * @param $name
     * @param $type
     * @param bool $nullable
     * @param string $default
     * @param string $extra
     * @return array
     */
    protected function new_column($name, $type, $nullable = false, $default = '', $extra = '') {
        return $this->column_from_field([
            'name' => $name,
            'type' => $type,
            'nullable' => $nullable,
            'default' => $default,
            'extra' => $extra,
        ]);
    }

    /**
     * Creates an index information array from a field information array.
     * @param $field
     * @return array
     */
    protected function index_from_field($field) {
        $name = '';
        if ($field['type'] == 'PRIMARY KEY') {
            $name = 'PRIMARY';
        } else if (array_key_exists('name', $field) && strlen($field['name'])) {
            $name = $field['name'];
        } else {
            $name = implode('-', $field['columns']);
        }
        return [
            'type' => $field['type'],
            'columns' => $field['columns'],
            'name' => $name,
        ];
    }

    /**
     * Creates an index information array from parameters.
     * @param $type
     * @param $columns
     * @param string $name
     * @return array
     */
    protected function new_index($type, $columns, $name = '') {
        return $this->index_from_field([
            'type' => $type,
            'columns' => $columns,
            'name' => $name,
        ]);
    }

    /**
     * @param $table_name
     * @return bool
     */
    public function post_create_table_sql($table_name) {
        return false;
    }

    /**
     * @param $table_name
     * @return bool
     */
    public function post_drop_table_sql($table_name) {
        return false;
    }

    /**
     * @param $table_name
     * @param $column_name
     * @return bool
     */
    public function post_create_column_sql($table_name, $column_name) {
        return false;
    }

    /**
     * @param $table_name
     * @param $column_name
     * @return bool
     */
    public function post_drop_column_sql($table_name, $column_name) {
        return false;
    }

    /**
     * @param $table_name
     * @param $column_name
     * @return bool
     */
    public function post_change_column_sql($table_name, $column_name) {
        return false;
    }

    /**
     * Is called when listing columns, override and return a list of column information arrays to prepend.
     * @param $table_name
     * @return array
     */
    protected function prepend_columns($table_name) {
        return [];
    }

    /**
     * Is called when listing indices, override and return a list of index information arrays to prepend.
     * @param $table_name
     * @return array
     */
    protected function prepend_indices($table_name) {
        return [];
    }

    /**
     * Is called when listing columns, override and return a list of column information arrays to append.
     * @param $table_name
     * @return array
     */
    protected function append_columns($table_name) {
        return [];    }

    /**
     * * Is called when listing indices, override and return a list of index information arrays to append.
     * @param $table_name
     * @return array
     */
    protected function append_indices($table_name) {
        return [];
    }

    public function is_summary_table($table_name) {
        $table = $this->table($table_name);
        if (array_key_exists('imported_from', $table)) return false;
        return true;
    }

    public function get_imported_from_field($table_name) {
        $fields = $this->fields($table_name);
        foreach ($fields as $field) {
            if (array_key_exists('imported_from', $field)) return $field['imported_from'];
        }
        return false;
    }

    public function import_files($table_name) {
        $table = $this->table($table_name);
        if (array_key_exists('imported_from', $table)) return $table['imported_from'];
        return [];
    }
}