<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 4/18/16
 * Time: 12:25 PM
 */

namespace Visionware\DataManager\Schema;

class LiveSchema extends Schema {
    protected function prepend_columns($table_name) {
        if (starts_with($table_name, 'z_')) {
            return [$this->new_column('id', 'binary(16)')];
        } else if ($this->has_uuid($table_name)) {
            return [$this->new_column('id', 'binary(16)')];
        } else if ($this->has_bigint_id($table_name)) {
            return [$this->new_column('id', 'bigint unsigned', false, '', 'auto_increment')];
        } else if ($this->has_int_id($table_name)) {
            return [ $this->new_column('id', 'int(11) unsigned', false, '', 'auto_increment'),];
        }
        return [];
    }

    public function post_create_table_sql($table_name) {
        if ($this->is_synced_down($table_name)) {
            $selectArray = array();
            $fields = $this->fields($table_name);
            foreach ($fields as $field) {
                $columnName = $field['name'];
                $cst_in_version = isset($field['cst_in_version']) ? $field['cst_in_version'] : 'all';
                if ($cst_in_version === 'all') {
                    if (starts_with($field['type'], "binary")) {
                        $selectArray[] = "UuidFromBin(`$columnName`) AS '$columnName'";
                    } else {
                        $selectArray[] = "`$columnName`";
                    }
                }
            }
            $selectString = implode(', ', $selectArray);

            return <<<SQL
CREATE VIEW `{$table_name}_v1` AS
SELECT $selectString
FROM `{$table_name}`;
SQL;
        }
        return false;
    }

    public function post_drop_table_sql($table_name) {
        if ($this->is_synced_down($table_name)) {
            return "DROP VIEW `{$table_name}_v1`;";
        }
        return false;
    }

    public function post_change_column_sql($table_name, $column_name) {
        return $this->post_drop_table_sql($table_name) . " " . $this->post_create_table_sql($table_name);
    }

    public function post_create_column_sql($table_name, $column_name) {
        return $this->post_change_column_sql($table_name, $column_name);
    }

    public function post_drop_column_sql($table_name, $column_name) {
        return $this->post_change_column_sql($table_name, $column_name);
    }

    protected function prepend_indices($table_name) {
        if (starts_with($table_name, 'z_')) return [];
        if ($this->pk_type($table_name) != 'none') return [ $this->new_index('PRIMARY KEY', ['id'], 'PRIMARY') ];
        return [];
    }

    protected function append_indices($table_name) {
        $indices = [];
        if ($this->is_synced_down($table_name)) {
            $indices[] = $this->new_index('KEY', ['date_uploaded']);
        }
        return $indices;
    }

    protected function append_columns($table_name) {
        $cols = [];
        if ($this->has_timestamps($table_name) || $this->is_synced_down($table_name)) {
            $cols[] = $this->new_column('date_created', 'datetime', true, 'CURRENT_TIMESTAMP');
            $cols[] = $this->new_column('date_modified','datetime', true,'CURRENT_TIMESTAMP', 'ON UPDATE CURRENT_TIMESTAMP');
        }
        if ($this->is_synced_down($table_name)) {
            $cols[] = $this->new_column('date_uploaded','datetime', true,'CURRENT_TIMESTAMP', 'ON UPDATE CURRENT_TIMESTAMP');
        }
        if ($this->has_deleted($table_name)) {
            $cols[] = $this->new_column('deleted_at', 'datetime', true, 'NULL');
        }
        return $cols;
    }
    
    public function tables() {
        $parent_tables = parent::tables();
        foreach ($parent_tables as $parent_table_name => $parent_table) {
            if ($this->is_synced_up($parent_table_name)) {
                $table_name = "z_" . $parent_table_name;
                $parent_tables[$table_name] = $this->table($table_name);
            }
        }
        return $parent_tables;
    }

    public function table($table_name) {
        if (starts_with($table_name, "z_")) {
            $parent_table_name = str_replace("z_", '', $table_name);
            return $this->z_table($parent_table_name, $table_name);
        }
        return parent::table($table_name);
    }

    public function z_table($parent_table_name, $table_name) {
        $fields = [
            $this->new_column('session_key', 'varchar(32)'),
            $this->new_column('batch_key', 'varchar(32)'),
        ];
        $fields = array_merge($fields, parent::fields($parent_table_name));
        $parent_indices = parent::indices($parent_table_name);
        $pkColumns = array();
        foreach ($parent_indices as $parent_index) {
            if ($parent_index['type'] == 'PRIMARY KEY') {
                $pkColumns = $parent_index['columns'];
                break;
            }
        }
        $indices = [
            $this->new_index('PRIMARY KEY', array_merge(['session_key'], $pkColumns), 'PRIMARY'),
            $this->new_index('KEY', ['session_key', 'batch_key'], 'session_batch_key_index'),
        ];
        return ['table' => $table_name, 'fields' => $fields, 'indices' => $indices];
    }
}