<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/13/16
 * Time: 10:16 AM
 */

namespace Visionware\DataManager;

use Closure;
use Visionware\DataManager\Facades\DataManager;
use Carbon\Carbon;
use Illuminate\Database\Connection;
use Visionware\DataManager\Exceptions\IngestionFieldsMismatchException;
use Visionware\DataManager\Exceptions\RemoteFileDownloadException;
use Visionware\DataManager\Exceptions\RemoteFileNotFoundException;
use Visionware\DataManager\Exceptions\UnableToCreateLoadFileException;

class Ingester extends DataManagerProcess {
    /**
     * @var Connection
     */
    protected $db;
    protected $snapshot_files;
    protected $new_files_count;

    public function __construct($schema) {
        parent::__construct($schema);
        $this->db = DataManager::history_db();
        $this->snapshot_files = [];
        $this->new_files_count = 0;
    }

    public function go() {
        $start = Carbon::now();
        if (!in_array($this->table_name, $this->schema->table_names())) {
            $this->error("Table {$this->table_name} is not defined!");
            return;
        }
        $import_files = $this->table_import_files();
        if (!is_array($import_files)) $import_files = [$import_files];
        $this->notice("Starting to ingest table " . $this->table_name);
        try {
            $this->db->beginTransaction();
            foreach ($import_files as $file_name) {
                    $this->info("Starting to ingest $file_name");
                    $this->ingest_file($file_name);
                    $this->info("Finished ingesting $file_name");
            }

            if ($this->new_files_count) {
                $this->info("Updating snapshots table from staging...");
                $this->update_snapshots();
            }
        } catch (\Exception $e) {
            $this->emergency("Caught exception while ingesting file $file_name for table {$this->table_name}! Rolling back...", ['exception' => $e]);
            $this->db->rollBack();
            return;
        }
        $this->db->commit();

        $diff = Carbon::now()->diffForHumans($start, true);
        $this->notice("Finished ingesting table " . $this->table_name . " in $diff total");
    }

    private function db_last_modified($file_name) {
        $db_last_modified = $this->db->table($this->table_name . '_files')->where('file_name', '=', $file_name)->max(
            'last_modified'
        );
        return $db_last_modified;
    }

    private function last_modified($file_name) {
        $disk = DataManager::import_source_disk();
        $path = DataManager::import_source_path() . $file_name;
        if (!$disk->exists($path)) {
            throw new RemoteFileNotFoundException("File $path not found, skipping...");
        }
        $last_modified_ut = $disk->lastModified($path);
        return date('Y-m-d H:i:s', $last_modified_ut);
    }

    private function ingest_file($file_name) {
        $last_modified = $this->last_modified($file_name);
        $db_last_modified = $this->db_last_modified($file_name);
        $is_modified = !($last_modified <= $db_last_modified);
        if (!$is_modified && !$this->force) {
            $file_id = $this->db->table("{$this->table_name}_files")->where('file_name', '=', $file_name)->where('last_modified', '=', $last_modified)->value('id');
            $snapshot_files[$file_id] = $last_modified;
            $this->info("File is up-to-date, skipping");
            return;
        }
        $this->info("Downloading file...");
        $local_path = $this->download_file($file_name);
        $this->info("Building LOAD DATA file...");
        $destination_columns = $this->destination_columns();
        $field_column_map = $this->field_column_map();
        $load_file_path = $local_path . '.load';
        $load_columns = $this->build_load_file($local_path, $load_file_path, $file_name, $destination_columns, $field_column_map, $last_modified);

        $this->info("Truncating staging table...");
        $this->db->table("{$this->table_name}_staging")->truncate();

        $this->info("Loading data into staging table...");
        $this->load_data($load_file_path, $load_columns);

        $this->info("Updating records table from staging...");
        $this->update_records($destination_columns);

        $this->info("Updating files table from staging...");
        $this->update_files();

        $file_id = $this->db->table("{$this->table_name}_files")->where('file_name', '=', $file_name)->where('last_modified', '=', $last_modified)->value('id');

        if ($this->force) {
            $this->info("Deleting old file_record rows...");
            $this->delete_old_records($file_id);
        }

        $this->info("Updating file_record table from staging...");
        $this->update_file_records();

        $this->snapshot_files[$file_id] = $last_modified;
        $this->new_files_count++;
    }

    private function destination_columns() {
        $destination_columns = [];
        foreach ($this->table_fields() as $field) {
            if (!array_key_exists('import_field', $field)) continue;
            $destination_columns[] = $field['name'];
        }
        return $destination_columns;
    }

    private function field_column_map() {
        $field_column_map = [];
        foreach ($this->table_fields() as $field) {
            if (!array_key_exists('import_field', $field)) continue;
            $field_column_map[$field['import_field']] = $field['name'];
        }
        return $field_column_map;
    }

    private function download_file($file_name) {
        $local_path = sys_get_temp_dir() . DIRECTORY_SEPARATOR ."$file_name";

        $this->info("Downloading file $file_name...");
        $disk = DataManager::import_source_disk();
        $remote_path = DataManager::import_source_path() . $file_name;
        if (false === @file_put_contents($local_path, $disk->get($remote_path))) {
            throw new RemoteFileDownloadException("Error " . error_get_last() . " while downloading $remote_path, skipping...");
        }
        return $local_path;
    }

    private function build_load_file($local_path, &$load_file_path, $file_name, $destination_columns, $field_column_map, $last_modified) {
        $local_file_object = new \SplFileObject($local_path, 'r');
        if ($local_file_object === false) {
            throw new UnableToCreateLoadFileException("Unable to open $local_path");
        }

        $local_file_object->setFlags(
            \SplFileObject::READ_AHEAD
            | \SplFileObject::SKIP_EMPTY
            | \SplFileObject::DROP_NEW_LINE
        );
        
        $local_file_object->seek(PHP_INT_MAX);
        $local_file_object->rewind();

        $source_fields = explode("\t", $local_file_object->current());
        $local_file_object->next();

        foreach ($field_column_map as $src => $dest) {
            if (!in_array($src, $source_fields)) {
                $this->emergency("Import file does not contain field $src", ['source_fields' => $source_fields, 'field_column_map' => $field_column_map]);
                throw new IngestionFieldsMismatchException("Import file does not contain field $src");
            }
        }
        if (count($source_fields) != count($destination_columns)) {
            $this->warn("Import file field count does not match definition!", ['source_fields' => $source_fields, 'destination_columns' => $destination_columns]);
        }

        $load_file_object = new \SplFileObject($load_file_path, 'w');
        $current_line = 0;
        while (!$local_file_object->eof()) {
            $output_row = [];
            $raw = explode("\t", $local_file_object->current());
            $input_row = @array_combine($source_fields, $raw);
            if ($input_row !== false) {
                foreach ($field_column_map as $field => $column) {
                    $output_row[$column] = str_replace([',', '"'], ['\\,', '\\"'], trim($input_row[$field]));
//                    if ($output_row[$column] == '') $output_row[$column] = '\N';
                }
                $special_fields = [
                    'record_hash' => md5(implode('', $output_row)),
                    'file_name' => $file_name,
                    'last_modified' => $last_modified,
                    'sequence' => $current_line,
                ];
                $output_row = array_merge($special_fields, $output_row);
                $load_file_object->fwrite(implode(',', $output_row) . "\n");
            } else {
                $this->warn("Line $current_line has invalid field count, skipping line.", ['raw_line' => $raw]);
            }

            $local_file_object->next();
            $current_line++;
        }
        $load_file_object->fflush();
        $sort_file_path = "$load_file_path.sorted";
        $output = $return = false;
        @exec("sort $load_file_path > $sort_file_path", $output, $return);
        if ($return === 0) $load_file_path = $sort_file_path;

        return implode(', ', array_keys($output_row));
    }

    private function load_data($load_file_path, $load_columns) {
        $sql =
            "LOAD DATA LOCAL INFILE '$load_file_path' IGNORE INTO TABLE `{$this->table_name}_staging` FIELDS TERMINATED BY ',' ($load_columns)";
        $this->debug($sql);
        $this->db->getPdo()->exec(str_replace("\\", "/", $sql));
    }

    private function update_records($destination_columns) {
        $columns_string = 'record_hash, ' . implode(', ', $destination_columns);
        $sql =
            "INSERT IGNORE INTO `{$this->table_name}_records` ($columns_string) SELECT $columns_string FROM `{$this->table_name}_staging`";
        $this->debug($sql);
        $this->db->statement($sql);
    }

    private function update_files() {
        $sql =
            "INSERT IGNORE INTO `{$this->table_name}_files` (file_name, last_modified) SELECT s.file_name, s.last_modified FROM `{$this->table_name}_staging` s LIMIT 1";
        $this->debug($sql);
        $this->db->statement($sql);
    }

    private function delete_old_records($file_id) {
        $sql = "DELETE FROM `{$this->table_name}_file_record` where file_id = $file_id";
        $this->debug($sql);
        $this->db->statement($sql);
    }

    private function update_file_records() {
        $sql =
            "INSERT IGNORE INTO `{$this->table_name}_file_record` (file_id, record_hash, sequence) SELECT f.id, s.record_hash, s.sequence FROM `{$this->table_name}_staging` s JOIN `{$this->table_name}_files` f USING (last_modified)";
        $this->debug($sql);
        $this->db->statement($sql);
    }

    private function update_snapshots() {
        $snapshot_last_modified = max(array_values($this->snapshot_files));
        $sql = "DELETE FROM `{$this->table_name}_snapshots` WHERE last_modified = \"$snapshot_last_modified\"";
        $this->debug($sql);
        $this->db->statement($sql);
        foreach ($this->snapshot_files as $file_id => $nothing) {
            $sql = "REPLACE INTO `{$this->table_name}_snapshots` (last_modified, file_id) VALUES (\"$snapshot_last_modified\", $file_id)";
            $this->debug($sql);
            $this->db->statement($sql);
        }
    }
}