<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/13/16
 * Time: 10:16 AM
 */

namespace Visionware\DataManager;

use Visionware\DataManager\Facades\DataManager;
use Carbon\Carbon;
use Illuminate\Database\Connection;
use Visionware\DataManager\Exceptions\IngestionFieldsMismatchException;
use Visionware\DataManager\Exceptions\RemoteFileDownloadException;
use Visionware\DataManager\Exceptions\RemoteFileNotFoundException;
use Visionware\DataManager\Exceptions\UnableToCreateLoadFileException;

class Importer extends DataManagerProcess {
    /**
     * @var Connection
     */
    protected $history_db;
    /**
     * @var Connection
     */
    protected $live_db;
    private $tables_to_import;
    private $iterations;
    private $source_table_name;
    private $live_table_name;
    private $history_schema_name;
    private $live_schema_name;
    private $is_summary;

    public function __construct($schema) {
        parent::__construct($schema);
        $this->history_db = DataManager::history_db();
        $this->live_db = DataManager::live_db();
        $this->history_schema_name = $this->history_db->getDatabaseName();
        $this->live_schema_name = $this->live_db->getDatabaseName();
        $this->tables_to_import = false;
    }

    public function tables_to_import($tables) {
        $this->tables_to_import = $tables;
        return $this;
    }

    public function go() {
        if ($this->tables_to_import === false) $this->tables_to_import = [$this->table_name];
        try {
            $this->live_db->beginTransaction();
            foreach ($this->tables_to_import as $table_name) {
                $this->table_name = $table_name;
                $start = Carbon::now();
                $this->notice("Starting to import table " . $this->table_name);
                $this->import_table();
                $diff = Carbon::now()->diffForHumans($start, true);
                $this->notice("Finished importing table " . $this->table_name . " in $diff total");
            }
            $this->live_db->commit();
        } catch (\Exception $e) {
            $this->error("Caught exception while importing table {$this->table_name}! Rolling back...", ['exception' => $e]);
            $this->live_db->rollBack();
        }
    }

    public function import_table() {
        if (!in_array($this->table_name, $this->schema->table_names())) {
            $this->error("Table {$this->table_name} is not defined!");
            return;
        }

        $this->is_summary = $this->table_is_summary_table();

        $this->source_table_name = $this->history_schema_name . '.' . $this->source_table_name() . '_latest';
        $this->live_table_name = $this->live_schema_name . '.' . $this->table_name;

        $source_last_modified = $this->source_last_modified();
        $last_updated = $this->live_last_updated();
        $is_modified = !($last_updated >= $source_last_modified);

        if (!$is_modified && !$this->force) {
            $this->info("Table {$this->table_name} is up-to-date, skipping");
            return;
        }

        $unique_key_columns = $this->table_table_key();
        $unique_key_columns_string = implode(', ', $unique_key_columns);

        $this->iterations = 1;

        $foreigners = $this->foreigners();

        $joins_result = $this->joins($foreigners, $unique_key_columns_string);
        $joins = $joins_result['joins'];
        $table_name_mapping = $joins_result['table_name_mapping'];

        $columns = $this->build_columns($table_name_mapping);

        $this->info(sprintf("Updating live table %s (%d %s)...", $this->live_table_name, $this->iterations, str_plural('iteration', $this->iterations)));
        $deleted_joins_string = $this->build_deleted_joins_string($unique_key_columns);
        $this->update_live($columns, $joins, $deleted_joins_string);

        $this->info('Updating last modified metadata record...');
        $this->update_metadata($source_last_modified);


    }

    private function source_table_name() {
        if ($this->is_summary) {
            return $this->table_get_imported_from_field();
        } else {
            return $this->table_name;
        }
    }

    private function source_last_modified() {
        if ($this->is_summary) {
            return $this->live_db->table($this->history_db->getDatabaseName() . '.' . $this->source_table_name() . '_files')->max('last_modified');
        } else {
            return $this->live_db->table($this->history_db->getDatabaseName() . '.' . $this->source_table_name() . '_files')->max('last_modified');
        }
    }

    private function live_last_updated() {
        return $this->live_db->table('datamanager_metadata')
            ->where('metadata_type', '=', 'last_updated')
            ->where('metadata_key', '=', $this->table_name)
            ->value('date_value')
        ;
    }

    private function foreigners() {
        $foreigners = [];
        foreach ($this->table_fields() as $field) {
            $letter = "A";
            if (array_key_exists('extra_joins', $field)) {
                foreach ($field['extra_joins'] as $extra_join) {
                    $fake = ['name' => $field['name']];
                    if (array_key_exists('imported_from', $extra_join)) $fake['imported_from'] = $extra_join['imported_from'];
                    if (array_key_exists('on_table', $extra_join)) $fake['on_table'] = $extra_join['on_table'];
                    if (array_key_exists('using', $extra_join)) $fake['using'] = $extra_join['using'];
                    if (array_key_exists('join_on', $extra_join)) $fake['join_on'] = $extra_join['join_on'];
                    $foreigners[$field['name'] . $letter++] = $fake;
                }
            }
            if (array_key_exists('foreign', $field)) {
                $foreigners[$field['name']] = $field;
                if ($field['on_table'] == $this->table_name) $this->iterations++;
            }
        }
        return $foreigners;
    }

    private function joins($foreigners, $unique_key_columns_string) {
        if ($this->is_summary) {
            return $this->summarized_joins($foreigners, $unique_key_columns_string);
        } else {
            return $this->imported_joins($foreigners, $unique_key_columns_string);
        }
    }

    private function imported_joins($foreigners, $unique_key_columns_string) {
        $joins = [];
        $joins[$this->table_name] = "LEFT OUTER JOIN {$this->live_table_name} USING ($unique_key_columns_string)";
        $table_name_mapping = [];
        $letter = "A";
        foreach ($foreigners as $index_key => $index) {
            $foreign_table_base = "{$this->live_schema_name}.$index[on_table]";
            $foreign_table = $index['on_table'];
            while (array_key_exists($foreign_table, $joins)) {
                $foreign_table =
                    $index['on_table'] . $letter++;
            }
            $join_ons = [];
            foreach ($index['join_on'] as $import_join_column) {
                $join_ons[] = $foreign_table
                    . '.'
                    . $import_join_column['foreign']
                    . ' = '
                    . $this->source_table_name
                    . '.'
                    . $import_join_column['local'];
                $table_name_mapping[$index_key] = $foreign_table;
            }
            $join_ons_string = implode(' AND ', $join_ons);

            $joins[$foreign_table] = "LEFT OUTER JOIN $foreign_table_base $foreign_table ON ($join_ons_string)";
        }
        return ['joins' => $joins, 'table_name_mapping' => $table_name_mapping];
    }

    private function summarized_joins($foreigners, $unique_key_columns_string) {
        $joins = [];
        $joins[$this->table_name] = "LEFT OUTER JOIN {$this->live_table_name} USING ($unique_key_columns_string)";
        $table_name_mapping = [];
        foreach ($foreigners as $index_key => $index) {
            if (array_key_exists('on_table', $index)) {
                $foreign_table_base = "{$this->live_schema_name}.$index[on_table]";
                $foreign_table = $index['on_table'];
            } else if (array_key_exists('imported_from', $index)) {
                $foreign_table_base = "{$this->history_schema_name}.$index[imported_from]_latest";
                $foreign_table = $index['imported_from'];
            }
            if (array_key_exists('join_on', $index)) {
                $join_ons = [];
                foreach ($index['join_on'] as $import_join_column) {
                    $join_ons[] = $foreign_table
                        . '.'
                        . $import_join_column['foreign']
                        . ' = '
                        . $this->source_table_name
                        . '.'
                        . $import_join_column['local'];
                }
                $join_ons_string = implode(' AND ', $join_ons);

                $joins[$foreign_table] = "LEFT JOIN $foreign_table_base $foreign_table ON ($join_ons_string)";
            } else if (array_key_exists('using', $index)) {
                $joins[$foreign_table] = "LEFT JOIN $foreign_table_base $foreign_table USING ($index[using])";
            }
            $table_name_mapping[$index_key] = $foreign_table;
        }
        return ['joins' => $joins, 'table_name_mapping' => $table_name_mapping];
    }

    private function build_columns($table_name_mapping) {
        $live_columns = [];
        $select_columns = [];
        if ($this->table_has_uuid()) {
            $live_columns = ['id'];
            $select_columns[] = "COALESCE({$this->live_table_name}.id, UuidToBin(UUID()))";
        }
        $live_updates = [];
        foreach ($this->table_fields() as $column) {
            if (array_key_exists('no_import', $column) && $column['no_import']) continue;

            $column_key = $column['name'];
            if (array_key_exists('foreign', $column)) {
                $select_columns[] = $table_name_mapping[$column_key] . ".$column[foreign] $column[name]";
            } else {
                if ($this->is_summary) {
                    $select_columns[] = $this->source_table_name . '.' . $column['name'];
                } else {
                    $select_columns[] = $this->import_transform($column, $this->source_table_name) . " $column[name]";
                }
            }

            $live_columns[] = $column['name'];
            $live_updates[] = "$column[name]=VALUES($column[name])";
        }
        $live_updates[] = "deleted_at=NULL";
        return [
            'live' => $live_columns,
            'select' => $select_columns,
            'update' => $live_updates,
        ];
    }

    private function update_live($columns, $joins, $deleted_joins_string) {
        $live_columns = $columns['live'];
        $select_columns = $columns['select'];
        $live_updates = $columns['update'];

        $select_columns_string = implode(', ', $select_columns);
        $live_columns_string = implode(', ', $live_columns);
        $live_updates_string = implode(', ', $live_updates);

        $joins_string = implode(" ", $joins);
        $count = 1;
        while ($this->iterations > 0) {
            $this->info("Inserting into live (iteration $count)...");
            $this->insert_live($live_columns_string, $select_columns_string, $joins_string, $live_updates_string);

            $this->info("Updating deleted records (iteration $count)...");
            $this->update_deleted($deleted_joins_string);
            $this->iterations--;
            $count++;
        }
    }

    private function insert_live($live_columns_string, $select_columns_string, $joins_string, $live_updates_string) {
        $sql =
            "INSERT IGNORE INTO {$this->live_table_name} ($live_columns_string) SELECT $select_columns_string FROM {$this->source_table_name} $joins_string ON DUPLICATE KEY UPDATE $live_updates_string;";
        $this->debug($sql);
        $this->live_db->affectingStatement($sql);
    }

    private function build_deleted_joins_string($unique_key_columns) {
        $joins = [];
        foreach ($unique_key_columns as $key_column) {
            $live_column = $this->table_field($key_column);
            $live_column_string = $this->import_transform($live_column, $this->live_table_name, true);
            $joins[] = $live_column_string . ' = ' . $this->source_table_name . '.' . $key_column;
        }
        return implode(' AND ', $joins);
    }

    private function update_deleted($deleted_joins_string) {
        $sql =
            "UPDATE {$this->live_table_name} LEFT OUTER JOIN {$this->source_table_name} ON ($deleted_joins_string) SET {$this->live_table_name}.deleted_at = NOW() WHERE {$this->source_table_name}.record_hash IS NULL AND {$this->live_table_name}.deleted_at IS NULL;";
        $this->debug($sql);
        $this->live_db->affectingStatement($sql);
    }

    private function update_metadata($source_last_modified) {
        $sql =
            "REPLACE INTO datamanager_metadata (`metadata_type`, `metadata_key`, `date_value`) VALUES ('last_updated', '{$this->table_name}', '$source_last_modified')";
        $this->debug($sql);
        $this->live_db->affectingStatement($sql);
    }
    private function import_transform($column, $table, $inverse = false) {
        if (!array_key_exists('transformation', $column)) return "$table.$column[name]";
        $transformation_type = $column['transformation']['type'];
        $method = $inverse ? "inverse_transform_$transformation_type" : "transform_$transformation_type";
        return call_user_func_array([$this, $method], [$table . '.' . $column['name'], $column]);
//        return str_replace('$$$', "$table.$column[name]", $column['transformation']);
    }

    private function transform_date($qualified_column, $column) {
        $format = $column['transformation']['format'];
        return "STR_TO_DATE($qualified_column, \"$format\")";
    }

    private function inverse_transform_date($qualified_column, $column) {
        $format = $column['transformation']['format'];
        return "IF($qualified_column = '0000-00-00', '', DATE_FORMAT($qualified_column, \"$format\"))";
    }

    private function transform_boolean($qualified_column, $column) {
        $true_value = $column['transformation']['true_value'];
        $false_value = $column['transformation']['false_value'];
        return "CASE WHEN $qualified_column = '$true_value' THEN 1 WHEN $qualified_column = '$false_value' THEN 0 END";
    }

    private function inverse_transform_boolean($qualified_column, $column) {
        $true_value = $column['transformation']['true_value'];
        $false_value = $column['transformation']['false_value'];
        return "CASE WHEN $qualified_column = 1 THEN '$true_value' WHEN $qualified_column = 0 THEN '$false_value' END";
    }

    private function transform_round($qualified_column, $column) {
        $value = $column['transformation']['value'];
        return "ROUND($qualified_column, $value)";
    }
}