<?php

namespace Visionware\DataManager\Console\Commands;

use Config;
use DataManager;
use DB;
use Exception;
use Illuminate\Console\Command;
use Storage;
use Visionware\DataManager\Grammars\MysqlGrammar;
use Visionware\DataManager\Grammars\SqliteEmptyGrammar;
use Visionware\DataManager\Schema\MySqlSchemaInfo;
use Visionware\DataManager\Schema\SqliteSchema;
use Visionware\DataManager\SchemaManager;

class CreateSchemaCommand extends Command {
    protected $signature = 'datamanager:create
        {--output= : Output to a file instead of stdout.}
        {--mode= : Creation mode history, live, sqlite, all (history, live) }
        {--recreate : Includes drop database statements}
    ';

    protected $description = 'Builds create statements';

    public function handle() {
        
        if ($this->option('mode') == 'sqlite' && is_null($this->option('output'))) {
            $this->error("You must specify an output file when using SQLite mode!");
            return false;
        }

        $modes = $this->option('mode') == 'all' ? ['history', 'live'] : [$this->option('mode')];

        $statements = [];
        $logger = DataManager::getLogger('creator', $this->getOutput());
        $definition = DataManager::definition();
        if ($this->option('mode') == 'sqlite') {
            $creator = new SchemaManager(new SqliteEmptyGrammar(), '', $logger, new SqliteSchema($definition), null);
            $creator->setRecreate();
            $statements = $creator->migrate($log);
        } else {
            foreach ($modes as $mode) {
                $connection = DataManager::db_for_mode($mode);
                $name = $connection->getDatabaseName();
                try {
                    $current_schema = new MySqlSchemaInfo($connection);
                } catch (Exception $e) {
                    $logger->notice("Failed to retrive current schema info with exception.", [$e]);
                    $current_schema = null;
                }
                $schema = DataManager::getSchema($mode);
                $creator = new SchemaManager(new MysqlGrammar(), $name, $logger, $schema, $current_schema);
                if ($this->option('recreate')) $creator->setRecreate();
                $statements = array_merge($statements, $creator->migrate());
            }
        }

        if (!count($statements)) {
            $logger->notice('No changes are necessary.');
        } else {
            if ($this->option('output')) {
                file_put_contents($this->option('output'), implode("\n\n", $statements));
            } else {
                print implode("\n\n", $statements);
            }
        }
    }
}