<?php

namespace Visionware\DataManager;

use Log;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Logger;
use Symfony\Bridge\Monolog\Formatter\ConsoleFormatter;
use Symfony\Bridge\Monolog\Handler\ConsoleHandler;
use Symfony\Component\Console\Output\OutputInterface;

trait HasCustomLoggingTrait {
    /** @var Logger */
    public $logger;
    protected $processName = '';
    protected $processType = 'datamanager';

    public function getConsoleFormat() {
        return "%start_tag%[%datetime%] %channel%.{$this->processType}-{$this->processName}.%level_name%:%end_tag% %message% %context% %extra%\n";
    }

    public function configureLogging($processName, $outputInterface = null) {
        $this->processName = $processName;
        $this->logger = new Logger(config('app.env'));

        $handler = new RotatingFileHandler(
            storage_path("/logs/datamanager/$processName.log"),
            7,
            Logger::DEBUG,
            true,
            0666
        );

        $this->logger->pushHandler($handler);

        if ($outputInterface) {
            $console_handler = new ConsoleHandler(
                $outputInterface, true, [
                    OutputInterface::VERBOSITY_NORMAL       => Logger::NOTICE,
                    OutputInterface::VERBOSITY_VERBOSE      => Logger::INFO,
                    OutputInterface::VERBOSITY_VERY_VERBOSE => Logger::INFO,
                    OutputInterface::VERBOSITY_DEBUG        => Logger::DEBUG,
                ]
            );
            $console_formatter = new ConsoleFormatter(['format' => $this->getConsoleFormat()]);
            $console_handler->setFormatter($console_formatter);
            $this->logger->pushHandler($console_handler);
        }
    }

    public function prepareLogContext($context) {
        $context['datamanager-process'] = $this->processName;
        return $context;
    }

    public function log($level, $message, $context) {
        $context = $this->prepareLogContext($context);
        if (strlen($context['datamanager-target'])) $message = '('.$context['datamanager-target'].') '.$message;
        if ($this->logger) $this->logger->log($level, $message, $context);
        return Log::log($level, $message, $context);
    }

    public function emergency($message, $context = []) {
        return $this->log('emergency', $message, $context);
    }

    public function alert($message, $context = []) {
        return $this->log('alert', $message, $context);
    }

    public function critical($message, $context = []) {
        return $this->log('critical', $message, $context);
    }

    public function error($message, $context = []) {
        return $this->log('error', $message, $context);
    }

    public function warning($message, $context = []) {
        return $this->log('warning', $message, $context);
    }

    public function notice($message, $context = []) {
        return $this->log('notice', $message, $context);
    }

    public function info($message, $context = []) {
        return $this->log('info', $message, $context);
    }

    public function debug($message, $context = []) {
        return $this->log('debug', $message, $context);
    }
}