<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/13/16
 * Time: 10:16 AM
 */

namespace Visionware\DataManager;

use DB;
use Exception;
use Illuminate\Support\Collection;
use Monolog\Logger;
use Visionware\DataManager\Definition\TableDefinition;
use Visionware\DataManager\Facades\DataManager;
use Carbon\Carbon;
use Illuminate\Database\Connection;

/**
 * Class Transferer
 * @package Visionware\DataManager
 * @method emergency(string $msg, array $context = [])
 * @method alert(string $msg, array $context = [])
 * @method critical(string $msg, array $context = [])
 * @method error(string $msg, array $context = [])
 * @method warning(string $msg, array $context = [])
 * @method notice(string $msg, array $context = [])
 * @method info(string $msg, array $context = [])
 */
class Transferer extends DataManagerProcess {
    /** @var Connection */
    protected $importDb;
    /** @var Connection */
    protected $historyDb;
    protected $noTransactions;
    protected $only;


    public function __construct($schema) {
        parent::__construct($schema);
        $this->importDb = DataManager::getImportConnection();
        $this->historyDb = DataManager::getHistoryImportConnection();
        $this->noTransactions = false;
        $this->only = false;
    }

    public function transferTable($tableName) {
        $tableName = $tableName . '_latest';
        $history_user = $this->historyDb->getConfig('username');
        $history_pass = $this->historyDb->getConfig('password');
        $history_host = $this->historyDb->getConfig('host');
        $history_schema = $this->historyDb->getConfig('database');

        $import_user = $this->importDb->getConfig('username');
        $import_pass = $this->importDb->getConfig('password');
        $import_host = $this->importDb->getConfig('host');
        $import_schema = $this->importDb->getConfig('database');

        $cmd = "mysqldump -u $history_user -p'{$history_pass}' -h $history_host -C $history_schema $tableName | mysql -u $import_user -p'{$import_pass}' -h $import_host $import_schema";
        $this->notice("Transferring dump of $tableName from historyimport...");
        $this->info($cmd);
        $output = [];
        exec($cmd, $output, $return_var);
        if ($return_var !== 0) {
            $this->emergency('Error while creating dump!', $output);
        }
    }

    public function go() {
        $allStart = Carbon::now();

        try {
            foreach ($this->schema->tablesToIngest() as $tableName) {
                $attempts = 3;
                $attempt = 0;
                while (++$attempt <= $attempts) {
                    try {
                        if ($this->only && !($tableName == $this->only)) continue;
                        $this->setTable($tableName->name());
                        $this->setLogPrefix($tableName->name());
                        $latestTable = "{$tableName}_latest";

                        $start = Carbon::now();

                        $this->notice("Transferring table from history to import... (attempt $attempt)");

                        $historyLastModified = $this->historyDb->table($latestTable)->value(DB::raw('MAX(date_modified)'));
                        $importLastModified = $this->importDb->table($latestTable)->value(DB::raw('MAX(date_modified)'));

                        $historyCount = $this->historyDb->table($latestTable)->count();
                        $importCount = $this->importDb->table($latestTable)->count();

                        $this->info("History max(date_modified): " . $historyLastModified);
                        $this->info("Import max(date_modified): " . $importLastModified);
                        $this->info("History record count: " . $historyCount);
                        $this->info("Import record count: " . $importCount);

                        if (
                            ($historyLastModified <= $importLastModified)
                            && ($historyCount == $importCount)
                            && !$this->force
                        ) {
                            $this->notice("Table is up to date, skipping");
                            continue 2;
                        }

                        $this->transferTable($tableName);
                        $importCount = $this->importDb->table($latestTable)->count();
                        $this->info("History record count: " . $historyCount);
                        $this->info("Import record count: " . $importCount);

                        if ($historyCount == $importCount) {
                            $this->info('Transfer is successful, renaming temp table to latest table...');
                        } else {
                            $this->emergency("Row counts do not match! Transfer operation failed!");
                            throw new Exception("Row counts do not match! Transfer operation failed!");
                        }

                        $end = Carbon::now();
                        $diff = $end->diffForHumans($start, true);
                        $this->notice("Transferred table in $diff total");
                        break;
                    } catch (\Exception $e) {
                        if ($attempt < $attempts) {
                            $this->error("Caught exception while transferring table! Trying again...", ['exception' => $e]);
                        } else {
                            throw $e;
                        }
                    }
                }
            }
        } catch (\Exception $e) {
            $this->error("Caught exception while transferring table! Rolling back...", ['exception' => $e]);
            if (!$this->noTransactions) $this->importDb->rollBack();
        }
        $diff = Carbon::now()->diffForHumans($allStart, true);
        $this->notice("Finished transfer process in $diff total");
    }

    public function noTransactions() {
        $this->noTransactions = true;
    }

    public function only($tableName) {
        $this->only = $tableName;
    }

}