<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 2/29/16
 * Time: 10:39 AM
 */

namespace Visionware\DataManager\Console\Commands;

use Carbon\Carbon;
use DB;
use Illuminate\Support\Facades\Storage;

class Importer extends DataManagerCommand {
    protected $datamanager_command_name = 'datamanager:import';
    protected $datamanager_signature = '
        {--ingest-only : Ingest only}
        {--import-only : Import only}
        {--force : Process even if database is up-to-date}
        {--skip= : Comma separated list of tables to skip}
    ';
    protected $description = 'Ingests and imports data';

    protected function datamanager_handle() {
        if (!$this->option('import-only')) {
            $this->ingest();
        }
        if (!$this->option('ingest-only')) {
            $this->import();
        }
    }

    protected function ingest() {
        $skip = explode(',', $this->option('skip'));
        $start = Carbon::now();
        foreach ($this->definition['import_order'] as $ingest_file) {
            $this->current_target = $ingest_file;
            if (in_array($ingest_file, $skip)) {
                $this->info('Skipping table per user override');
                continue;
            }
            $this->ingest_file($ingest_file);
        }
        $this->info("Ingestion complete in " . Carbon::now()->diffForHumans($start, true));
    }

    private function ingest_file($importName) {
        $start = Carbon::now();
        $current_definition = $this->definition['tables'][$importName];
        $table_name = $current_definition['name'];
        $db = DB::connection($this->config['history-database-connection']);
        if (is_array($current_definition['import_file'])) $import_files = $current_definition['import_file'];
        else $import_files = [$current_definition['import_file']];

        foreach ($import_files as $import_file_name) {
            $local_path = sys_get_temp_dir() . "/$import_file_name";
            $remote_path = $this->config['import_source.prepend_path'] . $import_file_name;
            $disk = Storage::disk($this->config['import_source.disk']);
            if (!$disk->exists($remote_path)) {
                $this->error("File $remote_path not found, skipping...");
                continue;
            }

            $this->info("Checking last modified date for $import_file_name...");
            $last_modified_ut = $disk->lastModified($remote_path);
            $last_modified = date('Y-m-d H:i:s', $last_modified_ut);

            try {
//                $db_last_modified = $db->selectOne("SELECT MAX(last_modified) lm FROM {$table_name}_files")->lm;
                $db_last_modified = $db->table($table_name . '_files')->max('last_modified');
            }  catch (\PDOException $e) {
                $this->error('PDOException while importing file! Skipping...', ['exception' => $e]);
                continue;
            }

            if (!$this->option('force') && $last_modified <= $db_last_modified) {
                $this->info("Table is up-to-date, skipping");
                continue;
            }

            $this->info("Downloading file $import_file_name...");
            if (false === @file_put_contents($local_path, $disk->get($remote_path))) {
                $this->error("Error while downloading $remote_path, skipping...", ['error' => error_get_last()]);
                continue;
            }

            $load_file_path = $local_path . '.load';

            $local_file_object = new \SplFileObject($local_path, 'r');
            if ($local_file_object === false) {
                $this->error("Unable to open $local_path");
                continue;
            }
//            $local_file_object->setCsvControl("\t", '');
            $local_file_object->setFlags(
                \SplFileObject::READ_AHEAD
                | \SplFileObject::SKIP_EMPTY
                | \SplFileObject::DROP_NEW_LINE
            );
            $local_file_object->seek(PHP_INT_MAX);
            $line_count = $local_file_object->key();
            $this->info("Opening file $local_path ($line_count lines)...");
            $local_file_object->rewind();

            $source_fields = explode("\t", $local_file_object->current());
            $local_file_object->next();

            $destination_columns = [];
            $field_column_map = [];
            foreach ($current_definition['columns'] as $column) {
                if (!isset($column['import_field'])) continue;
                $destination_columns[] = $column['name'];
                $field_column_map[$column['import_field']] = $column['name'];
            }

            if (count($source_fields) != count($destination_columns)) {
                $this->error("Import file field count does not match definition!", ['source_fields' => $source_fields, 'destination_columns' => $destination_columns]);
                continue;
            }

            $values = [];
            $this->info("Building LOAD DATA file...");
            $load_file_object = new \SplFileObject($load_file_path, 'w');
            $current_line = 0;
            while (!$local_file_object->eof()) {
                $output_row = [];
                $raw = explode("\t", $local_file_object->current());
                if ($values === false) return; //return on error
                $input_row = @array_combine($source_fields, $raw);
                if ($input_row !== false) {
                    foreach ($field_column_map as $field => $column) {
                        $output_row[$column] = str_replace([',', '"'], ['\\,', '\\"'], trim($input_row[$field]));
                    }
                    $output_row['record_hash'] = md5(implode('', $output_row));
//                    $this->debug('Hashing record...', ['hash' => $output_row['record_hash'], 'record' => implode('', $output_row)]);
                    $output_row['last_modified'] = $last_modified;
                    $output_row['sequence'] = $current_line;

                    $load_file_object->fwrite(implode(',', $output_row) . "\n");
                } else {
                    $this->warn("Line $current_line has invalid field count, skipping line.", ['raw_line' => $raw]);
                }

                $local_file_object->next();
                $current_line++;
            }
            $load_file_object->fflush();
            $columns_string = implode(', ', array_keys($output_row));
            $sql = "LOAD DATA LOCAL INFILE '$load_file_path' IGNORE INTO TABLE `{$table_name}_staging` FIELDS TERMINATED BY ',' ($columns_string)";
            try {
                $this->info("Loading data into staging table...");
                $this->debug($sql);
                $db->table("{$table_name}_staging")->truncate();
                $db->getPdo()->exec($sql);
                $db->beginTransaction();
                $columns_string = 'record_hash, ' . implode(', ', $destination_columns);
                $this->info("Updating records table from staging...");
                $sql = "INSERT IGNORE INTO `{$table_name}_records` ($columns_string) SELECT $columns_string FROM `{$table_name}_staging`";
                $this->debug($sql);
                $db->statement($sql);

                $this->info("Updating files table from staging...");
                $sql = "INSERT IGNORE INTO `{$table_name}_files` (last_modified) SELECT s.last_modified FROM `{$table_name}_staging` s LIMIT 1";
                $this->debug($sql);
                $db->statement($sql);

                $this->info("Updating file_record table from staging...");
                $sql = "INSERT IGNORE INTO `{$table_name}_file_record` (file_id, record_hash, sequence) SELECT f.id, s.record_hash, s.sequence FROM `{$table_name}_staging` s JOIN `{$table_name}_files` f USING (last_modified)";
                $this->debug($sql);
                $db->statement($sql);

                $db->commit();
            } catch (\PDOException $e) {
                $this->error('Exception while importing file! Skipping...', ['exception' => $e]);
                continue;
            }
            $this->notice("Imported file $local_path in " . Carbon::now()->diffForHumans($start, true) . " total\n");
        }
    }

    protected function import() {
        $start = Carbon::now();
        $history = DB::connection($this->config['history-database-connection'])->getDatabaseName();
        $live = DB::connection($this->config['live-database-connection'])->getDatabaseName();
        $db = DB::connection($this->config['live-database-connection']);

        $to_import_count = count($this->definition['import_order']);
        $imported_count = 0;
        $db->beginTransaction();
        try {
            foreach ($this->definition['import_order'] as $table_name) {
                /** 1) Get most recent data from history table and load it into the staging table
                 *  2) Populate the staging table with foreign key ID values by using the key fields defined
                 *  3) Update the live tables with the filled-in staging table values
                 */
                $imported_count++;

                $tbl_start = Carbon::now();
                $this->current_target = $table_name;
                $this->info("Importing table ($imported_count of $to_import_count)...");

                $definition = $this->definition['tables'][$table_name];

                $source_table = $history . '.' . $table_name . "_latest";
                $get_date_table = $history . '.' . $table_name . "_files";
                $live_table = $live . '.' . $table_name;

                $source_last_modified = $db->table($get_date_table)->max('last_modified');
                $last_updated =
                    $db->table('datamanager_metadata')
                        ->where('metadata_type', '=', 'last_updated')
                        ->where('metadata_key', '=', $table_name)
                        ->value('date_value');
                if (!$this->option('force') && $last_updated >= $source_last_modified) {
                        $this->info("Table contains most recent data already, skipping", ['last_modified' => $source_last_modified, 'last_updated' => $last_updated]);
                        continue;
                }
                $unique_key = $definition['indices']['unique_key']['columns'];
                uasort(
                    $unique_key,
                    function ($a, $b) {
                        return $a['sequence'] - $b['sequence'];
                    }
                );
                $unique_key_columns = [];
                foreach ($unique_key as $xx => $yy) $unique_key_columns[] = $yy['name'];
                $unique_key_columns_string = implode(', ', $unique_key_columns);

                $joins = [];
                $joins[$table_name] = "LEFT OUTER JOIN $live_table USING ($unique_key_columns_string)";
                $table_name_mapping = [];
                $letter = "A";
                foreach ($definition['indices'] as $index_key => $index) {
                    if ($index['index_type'] != 'foreign_key'
                        && !array_key_exists(
                            'import_join_columns',
                            $index
                        )
                    ) {
                        continue;
                    }
                    $foreign_table_base = "$live.$index[foreign_table]";
                    $foreign_table = $index['foreign_table'];
                    while (array_key_exists($foreign_table, $joins)) {
                        $foreign_table =
                            $index['foreign_table'] . $letter++;
                    }
                    uasort(
                        $index['import_join_columns'],
                        function ($a, $b) {
                            return $a['sequence'] - $b['sequence'];
                        }
                    );
                    $join_ons = [];
                    foreach ($index['import_join_columns'] as $import_join_column) {
                        $join_ons[] = $foreign_table
                            . '.'
                            . $import_join_column['foreign']
                            . ' = '
                            . $source_table
                            . '.'
                            . $import_join_column['local'];
                    }
                    $join_ons_string = implode(' AND ', $join_ons);

                    $joins[$foreign_table] = "LEFT OUTER JOIN $foreign_table_base $foreign_table ON ($join_ons_string)";
                    $table_name_mapping[$index['foreign_table']] = $foreign_table;
                }
                $joins_string = implode(" ", $joins);

                $select_columns = [];
                $live_columns = [];
                $live_updates = [];
                foreach ($definition['columns'] as $column_key => $column) {
                    if (!in_array($column['column_type'], ['primary', 'column', 'foreign_key', 'foreign_id', 'deleted_flag'])) continue;
                    if ($column['column_type'] == 'primary') {
                        $select_columns[] = "COALESCE($live_table.id, UuidToBin(UUID()))";
                    } else if ($column['column_type'] == 'deleted_flag') {
                        $select_columns[] = "NULL";
                    } else {
                        if ($column['column_type'] == 'foreign_id') {
                            $select_columns[] =
                                $table_name_mapping[$column['references_on']]
                                . ".$column[references_column] $column[name]";
                        } else {
                            $select_columns[] = $this->import_transform($column, $source_table) . " $column[name]";
                        }
                    }

                    $live_columns[] = $column['name'];

                    if ($column['column_type'] == 'deleted_flag') $live_updates[] = "$column[name]=NULL";
                    else if ($column['column_type'] != 'primary') $live_updates[] = "$column[name]=VALUES($column[name])";

                }

                $select_columns_string = implode(', ', $select_columns);
                $live_columns_string = implode(', ', $live_columns);
                $live_updates_string = implode(', ', $live_updates);

                $this->info("Updating live table $live_table...");
                $sql =
                    "INSERT IGNORE INTO $live_table ($live_columns_string) SELECT $select_columns_string FROM $source_table $joins_string ON DUPLICATE KEY UPDATE $live_updates_string;";
                $this->debug($sql);
                $db->affectingStatement($sql);

                $this->info('Updating deleted records...');
                $sql = "UPDATE $live_table LEFT OUTER JOIN $source_table USING ($unique_key_columns_string) SET $live_table.deleted_at = NOW() WHERE $source_table.record_hash IS NULL AND $live_table.deleted_at IS NULL;";
                $this->debug($sql);
                $db->affectingStatement($sql);

                $this->info('Updating last modified metadata record...');
                $sql = "REPLACE INTO datamanager_metadata (`metadata_type`, `metadata_key`, `date_value`) VALUES ('last_updated', '$table_name', '$source_last_modified')";
                $this->debug($sql);
                $db->affectingStatement($sql);

                $this->notice("Completed importing table in " . Carbon::now()->diffForHumans($tbl_start, true));
            }
        } catch (Exception $e) {
            $db->rollBack();
            $this->error('Exception caught', ['exception' => $e]);
            return;
        }
        $this->info("Committing...");
        $db->commit();
        $this->info("Import complete in " . Carbon::now()->diffForHumans($start, true));
    }

    private function import_transform($column, $table) {
        if (!array_key_exists('import_transformation', $column)) return "$table.$column[name]";
        return str_replace('$$$', "$table.$column[name]", $column['import_transformation']);
    }
}