<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/20/16
 * Time: 10:23 AM
 */

namespace Visionware\DataManager\Info;

use Closure;
use Illuminate\Support\Collection;
use Monolog\Logger;
use Visionware\DataManager\Actions\ActionManager;
use Visionware\DataManager\Definition\Definition;
use Visionware\DataManager\Definition\SchemaDefinition;
use Visionware\DataManager\Definition\TableDefinition;
use Visionware\DataManager\DeltaManager;
use Visionware\DataManager\Grammars\Grammar;
use Visionware\DataManager\Info\ColumnInfo;
use Visionware\DataManager\Info\TableInfo;

class SchemaInfo extends Info {
    /**
     * @var Collection
     */
    private $tables;

    /**
     * @var Collection
     */
    private $views;
    
    /**
     * @var ActionManager
     */

    private $actions;

    private $schemaName;

    protected function __construct() {
        parent::__construct();
        $this->tables = new Collection;
        $this->views = new Collection;
        $this->actions = new ActionManager;
        $this->actions = $this->createMetadataTable($this->actions);
        $this->actions = $this->initActions($this->actions);
        $this->schemaName = '';
    }

    public function schemaName() {
        return $this->schemaName;
    }

    public function setSchemaName($name) {
        $this->schemaName = $name;
    }

    private function createMetadataTable(ActionManager $actions) {
        $actions->schema()->appendTable(function (SchemaDefinition $schemaDefinition, SchemaInfo $schemaInfo) {
            $table = TableInfo::create('datamanager_metadata');
            $table->putColumn(ColumnInfo::create('metadata_type', 'varchar(255)'));
            $table->putColumn(ColumnInfo::create('metadata_key', 'varchar(255)'));
            $table->putColumn(ColumnInfo::create('int_value', 'int(11)', true, 'null'));
            $table->putColumn(ColumnInfo::create('text_value', 'varchar(255)', true, 'null'));
            $table->putColumn(ColumnInfo::create('longtext_value', 'longtext', true, 'null'));
            $table->putColumn(ColumnInfo::create('date_value', 'datetime', true, 'null'));
            $table->putIndex(IndexInfo::create('PRIMARY KEY', ['metadata_type', 'metadata_key']));
            return $table;
        });
        return $actions;
    }

    public static function createFromDefinition(SchemaDefinition $definition) {
        $instance = new static;
        $instance->tables = new Collection;
        $instance->putOther('definition', $definition);
        foreach ($definition->tables() as $tableDefinition) {
            $tableInfo = TableInfo::createFromDefinition($tableDefinition);
            $instance->putTable($tableInfo);
        }
        foreach ($definition->views() as $viewDefinition) {
            $viewInfo = ViewInfo::createFromDefinition($viewDefinition);
            $instance->putView($viewInfo);
        }
        $instance->executeActions();
        return $instance;
    }

    public function definition() {
        return $this->getOther('definition');
    }

    public function executeActions() {
        $this->actions->executeSchema($this->definition(), $this);
        foreach ($this->tables() as $table) {
            if (!$table->isDerived()) {
                $this->actions->executeTable($table->definition(), $table);
            }
        }
        foreach ($this->tables() as $table) {
            if (!$table->isDerived()) {
                $this->actions->executeTableView($table->definition(), $table);
            }
        }
    }

    public function tables() {
        return $this->tables->sortBy(function ($item) {
            return $item->name();
        });
    }

    public function views() {
        return $this->views->sortBy(function ($item) {
            return $item->name();
        });
    }

    public function putTable(TableInfo $table) {
        $table->setSchema($this);
        $this->tables->put($table->name(), $table);
    }

    public function prependTable(TableInfo $table) {
        $table->setSchema($this);
        $this->tables->prepend($table, $table->name());
    }

    public function appendTable(TableInfo $table) {
        $this->putTable($table);
    }

    public function removeTable($table_name) {
        if ($table_name instanceof TableInfo) $table_name = $table_name->name();
        $this->tables->offsetUnset($table_name);
    }

    public function putView(ViewInfo $view) {
        $view->setSchema($this);
        $this->views->put($view->name(), $view);
    }

    public function appendView(ViewInfo $view) {
        $this->putView($view);
    }

    public function toSql(Grammar $grammar) {
        $deltas = new DeltaManager;
        
        foreach ($this->tables() as $table) $deltas->createTable($table);
        foreach ($this->views() as $view) $deltas->createView($view);
        return $deltas->compile($grammar);
    }

    /**
     * @param SchemaInfo $currentSchema
     * @return DeltaManager
     */
    public function migrateFrom(SchemaInfo $currentSchema, Grammar $grammar, Logger $logger) {
        $deltas = new DeltaManager;

        foreach ($this->tablesToAdd($currentSchema) as $table) $deltas->createTable($table);
        foreach ($this->tablesToDrop($currentSchema) as $table) $deltas->dropTable($table);

        foreach ($this->tablesToCompare($currentSchema) as $table) {
            $currentTable = $currentSchema->tables()->get($table->name());
            foreach ($table->columnsToAdd($currentTable) as $column) $deltas->createColumn($column);
            foreach ($table->indicesToAdd($currentTable) as $index) $deltas->createIndex($index);
            foreach ($table->columnsToDrop($currentTable) as $column) {
                $deltas->dropColumn($column);
            }
            foreach ($table->indicesToDrop($currentTable) as $index) $deltas->dropIndex($index);
            foreach ($table->columnsToChange($currentTable) as $column) $deltas->alterColumn($column);
            foreach ($table->indicesToChange($currentTable) as $index) $deltas->alterIndex($index);
        }

        foreach ($this->viewsToAdd($currentSchema) as $view) $deltas->createView($view);
        foreach ($this->viewsToDrop($currentSchema) as $view) $deltas->dropView($view);
        foreach ($this->viewsToCompare($currentSchema) as $view) {
            if (!$view->compareTo($currentSchema->views()->get($view->name()))) $deltas->alterView($view);
        }
        return $deltas->compile($grammar, $logger);
    }

    private function tablesToAdd(SchemaInfo $current) {
        return $this
            ->tables()
            ->only(
                array_diff(
                    $this->tables()->keys()->toArray(),
                    $current->tables()->keys()->toArray()
                )
            )
        ;
    }

    private function tablesToDrop(SchemaInfo $current) {
        return $current
            ->tables()
            ->only(
                array_diff(
                    $current->tables()->keys()->toArray(),
                    $this->tables()->keys()->toArray()
                )
            )
        ;
    }

    private function tablesToCompare(SchemaInfo $current) {
        return $this
            ->tables()
            ->only(
                array_intersect(
                    $this->tables()->keys()->toArray(),
                    $current->tables()->keys()->toArray()
                )
            )
        ;
    }

    private function viewsToAdd(SchemaInfo $current) {
        return $this
            ->views()
            ->only(
                array_diff(
                    $this->views()->keys()->toArray(),
                    $current->views()->keys()->toArray()
                )
            )
            ;
    }

    private function viewsToDrop(SchemaInfo $current) {
        return $current
            ->views()
            ->only(
                array_diff(
                    $current->views()->keys()->toArray(),
                    $this->views()->keys()->toArray()
                )
            )
            ;
    }

    private function viewsToCompare(SchemaInfo $current) {
        return $this
            ->views()
            ->only(
                array_intersect(
                    $this->views()->keys()->toArray(),
                    $current->views()->keys()->toArray()
                )
            )
            ;
    }

    public function tablesToIngest() {
        return $this
            ->tables()
            ->filter(function ($item, $key) {
                return $item->shouldIngest();
            })->sortBy(function ($item, $key) {
                if ($key == 'PriceLists') return 'ZZZZZZZZZZZZZ';
                return $key;
            })
        ;
    }

    public function tablesToImport() {
        /** @var TableInfo $table */
        foreach ($this->tables() as $table) $table->buildDependencies();
        $resolved = new Collection;
        foreach ($this->tables() as $table) {
            if (!$table->shouldImport()) continue;
            $table->resolveDependencies($resolved);
//            print "\n\n";
        }
        return $resolved;
    }
    
    public function name() { return ''; }

    public function __toString() {
        return "";
    }

    public function __toDebugArray() {
        return [];
    }

    protected function initActions(ActionManager $actionManager) {
        return $actionManager;
    }
}