<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 1/27/16
 * Time: 11:47 AM
 */

namespace Visionware\DataManager\Console\Commands;

use Carbon\Carbon;
use Config;
use Illuminate\Console\Command;
use Illuminate\Log\Writer;
use Illuminate\Support\Facades\Mail;
use InvalidArgumentException;
use Log;
use Monolog\Formatter\LineFormatter;
use Monolog\Formatter\LogstashFormatter;
use Monolog\Handler\BufferHandler;
use Monolog\Handler\FingersCrossed\ErrorLevelActivationStrategy;
use Monolog\Handler\FingersCrossedHandler;
use Monolog\Handler\PHPConsoleHandler;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\SlackHandler;
use Monolog\Handler\StreamHandler;
use Monolog\Handler\SwiftMailerHandler;
use Monolog\Logger;
use Storage;
use Swift_Message;
use Symfony\Bridge\Monolog\Formatter\ConsoleFormatter;
use Symfony\Bridge\Monolog\Handler\ConsoleHandler;
use Symfony\Component\Console\Output\OutputInterface;
use Visionware\DataManager\DefinitionValidator;
use Visionware\DataManager\VisionwareSlackHandler;
use Visionware\Utilities\Console\Commands\PrependsDurationTrait;
use Visionware\Utilities\Console\Commands\PrependsOutputTrait;

abstract class DataManagerCommand extends Command {

    protected $config;
    protected $definition;
    protected $definitionObject;
    protected $signature = "";
    protected $datamanager_signature;
    protected $datamanager_command_name;
    protected $current_target;
    protected $last_duration;

    /**
     * @var Logger
     */
    protected $logger;
    private $buffer_handler;

//    protected $errors = [];

    public function __construct() {
        $this->signature = $this->datamanager_command_name
            . ' ' . $this->datamanager_signature
            . ' {--definition= : The name of the definition in the datamanager.php config file instead of default} '
            . ' {--skip-validation : Doesn\'t validate the schema JSON first}';
        parent::__construct();
        $this->last_duration = Carbon::now();
    }

    public function get_current_target() { return $this->current_target; }

    public function processRecord($record) {
        $record['extra']['current_definition'] = $this->definition['title'];
        $record['extra']['current_target'] = $this->current_target;
        $record['extra']['last_duration'] = Carbon::now()->diffForHumans($this->last_duration, true);
        $this->last_duration = Carbon::now();

        return $record;
    }

    public function handle() {
        $useDefinition = $this->option('definition');
        if (!strlen($useDefinition)) $useDefinition = Config::get('datamanager.default_definition');
        if (!strlen($useDefinition)) throw new InvalidArgumentException('No definition specified!');
        $this->config = array_dot(Config::get('datamanager.definitions.' . $useDefinition));
        $this->definition = json_decode(Storage::disk($this->config['disk'])->get($this->config['path']), true);
        if (!$this->option('skip-validation')) {
            $errors = DefinitionValidator::validate($this->definition);
            if (count($errors)) {
                foreach ($errors as $msg) $this->error($msg);

                return false;
            }
        }
        $this->initialize_loggers();
        $this->datamanager_handle();
        $this->buffer_handler->close();
    }

    private function initialize_loggers() {
        $this->logger = new Logger($this->datamanager_command_name);
        $this->logger->pushProcessor([$this, 'processRecord']);
        $date_format = "c";
        $output =
            "[%datetime%] (+%extra.last_duration%) %level_name% %channel%->%extra.current_definition%->%extra.current_target%: %message% %context%";

        $console_handler = new ConsoleHandler(
            $this->getOutput(), true, [
                OutputInterface::VERBOSITY_NORMAL       => Logger::NOTICE,
                OutputInterface::VERBOSITY_VERBOSE      => Logger::INFO,
                OutputInterface::VERBOSITY_VERY_VERBOSE => Logger::DEBUG,
                OutputInterface::VERBOSITY_DEBUG        => Logger::DEBUG,
            ]
        );
        $console_handler->setFormatter(new ConsoleFormatter("%start_tag%$output%end_tag%\n", $date_format));
        $this->logger->pushHandler($console_handler);

//        $file_handler = new RotatingFileHandler(storage_path('/logs/datamanager.log'), 7, Logger::DEBUG, true);
//
//        $file_handler->setFormatter(new LineFormatter("$output\n", $date_format));
//        $this->logger->pushHandler($file_handler);
//
//        $mail_handler = new SwiftMailerHandler(
//            Mail::getSwiftMailer(),
//            Swift_Message::newInstance("Data Manager Error Log - " . $this->definition['title'])
//                ->setFrom(env('MAIL_FROM'))
//                ->setTo(env('MAIL_ERRORS_TO')),
//            Logger::DEBUG
//        );
//        $mail_handler->setFormatter(new LineFormatter("$output\n"));
//
//        $this->buffer_handler = new BufferHandler($mail_handler);
//        $error_handler = new FingersCrossedHandler(
//            $this->buffer_handler,
//            new ErrorLevelActivationStrategy(Logger::ERROR)
//        );
//        $this->logger->pushHandler($error_handler);
//
//        $slack_handler = new VisionwareSlackHandler(
//            env('SLACK_API_TOKEN'),
//            env('SLACK_CHANNEL'),
//            env('SLACK_NICK'),
//            true,
//            'giant_head',
//            Logger::ERROR
//        );
//        $this->logger->pushHandler($slack_handler);
//
//        $logstash_formatter = new LogstashFormatter('datamanager-log', null, null, 'ctxt_', LogstashFormatter::V1);
//        $ls_handler = new RotatingFileHandler(storage_path('/logs/datamanager-ls.log'), 7, Logger::DEBUG, true);
//        $ls_handler->setFormatter($logstash_formatter);
//        $this->logger->pushHandler($ls_handler);
    }

    /**
     * Overrides the default Command handle method so that we can do the config/definition loading
     * @return mixed
     */
    abstract protected function datamanager_handle();

    public function debug($string, $context = []) {
        $this->logger->debug($string, $context);
        Log::debug($string, $context);
    }

    public function info($string, $context = []) {
        $this->logger->info($string, $context);
        Log::info($string, $context);
    }

    public function notice($string, $context = []) {
        $this->logger->notice($string, $context);
        Log::notice($string, $context);
    }

    public function warn($string, $context = []) {
        $this->logger->warn($string, $context);
        Log::warn($string, $context);
    }

    public function error($string, $context = []) {
        $this->logger->error($string, $context);
        Log::error($string, $context);
    }

}