<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 4/18/16
 * Time: 11:09 AM
 */

namespace Visionware\DataManager\Schema;

class Schema {
    private $definition;

    public function __construct($definition) {
        $this->definition = $definition;
    }

    public function definition() {
        return $this->definition;
    }

    protected function filter_table($table_name, $table) {
        return true;
    }

    public function tables() {
        $tables = [];
        foreach ($this->definition['tables'] as $table) {
            if (!$this->filter_table($table['table'], $table)) continue;
            $tables[$table['table']] = $table;
        }
        return $tables;
    }

    public function table_names() {
        $names = [];
        foreach ($this->tables() as $table_name => $table) {
            $names[] = $table_name;
        }
        return $names;
    }

    public function table($table_name) {
        return $this->definition['tables'][$table_name];
    }

    public function fields($table_name, $with_info = true) {
        $table = $this->table($table_name);
        $fields = [];
        foreach ($table['fields'] as $field) {
            if ($with_info) $fields[$field['name']] = $field;
            else $fields[] = $field['name'];
        }
        return $fields;
    }

    public function columns($table_name) {
        $columns = $this->prepend_columns($table_name);
        foreach ($this->fields($table_name) as $field_name => $field) {
            $columns[] = $this->column_from_field($field);
        }
        return array_merge($columns, $this->append_columns($table_name));
    }

    public function column($table_name, $column_name) {
        foreach ($this->columns($table_name) as $column) {
            if ($column['name'] == $column_name) return $column;
        }
        return false;
    }

    public function column_names($table_name) {
        $names = [];
        foreach ($this->columns($table_name) as $field) {
            $names[] = $field['name'];
        }
        return $names;
    }

    public function index_names($table_name) {
        $names = [];
        foreach ($this->indices($table_name) as $field) $names[] = $this->index_name($field);
        return $names;
    }

    public function index_name($field) {
        if (array_key_exists('name', $field) && strlen($field['name'])) return $field['name'];
        return implode('-', $field['columns']);
    }

    public function index($table_name, $index_name) {
        foreach ($this->indices($table_name) as $index) {
            if ($this->index_name($index) == $index_name) return $index;
        }
        return false;
    }

    public function indices($table_name) {
        $indices = $this->prepend_indices($table_name);
        if (count($this->table_key($table_name))) {
            $indices[] = $this->new_index('UNIQUE KEY', $this->table_key($table_name), 'unique_key');
        }
        foreach ($this->index_fields($table_name) as $index_name => $index) {
            $indices[] = $this->index_from_field($index);
        }
        foreach ($this->fields($table_name) as $field_name => $field) {
            if (array_key_exists('foreign', $field)) {
                $indices[] = $this->new_index('KEY', [$field_name], $field_name);
            }
        }
        return array_merge($indices, $this->append_indices($table_name));
    }

    public function table_key($table_name) {
        return array_key_exists('key', $this->table($table_name)) ? $this->table($table_name)['key'] : [];
    }

    public function index_fields($table_name, $with_info = true) {
        $table = $this->table($table_name);
        $indices = [];
        if (array_key_exists('indices', $table)) {
            foreach ($table['indices'] as $index) {
                if ($with_info) $indices[$this->index_name($index)] = $index;
                else $indices[] = $this->index_name($index);
            }
        }
        return $indices;
    }

    public function has_uuid($table_name) {
        return array_key_exists('has_uuid', $this->table($table_name)) && $this->table($table_name)['has_uuid'];
    }

    public function has_timestamps($table_name) {
        return !(array_key_exists('no_timestamps', $this->table($table_name)) && $this->table($table_name)['no_timestamps']);
    }

    public function has_deleted($table_name) {
        return !(array_key_exists('no_deleted', $this->table($table_name)) && $this->table($table_name)['no_deleted']);
    }

    protected function column_from_field($field) {
        return [
            'name' => $field['name'],
            'type' => $field['type'],
            'nullable' => array_key_exists('nullable', $field) ? $field['nullable'] : false,
            'default' => array_key_exists('default', $field) ? $field['default'] : '',
            'extra' => array_key_exists('extra', $field) ? $field['extra'] : '',
        ];
    }

    protected function new_column($name, $type, $nullable = false, $default = '', $extra = '') {
        return $this->column_from_field([
            'name' => $name,
            'type' => $type,
            'nullable' => $nullable,
            'default' => $default,
            'extra' => $extra,
        ]);
    }

    protected function index_from_field($field) {
        return [
            'type' => $field['type'],
            'columns' => $field['columns'],
            'name' => array_key_exists('name', $field) ? $field['name'] : '',
        ];
    }

    protected function new_index($type, $columns, $name = '') {
        return $this->index_from_field([
            'type' => $type,
            'columns' => $columns,
            'name' => $name,
        ]);
    }

    public function post_create_table_sql($table_name) {
        return false;
    }

    public function post_drop_table_sql($table_name) {
        return false;
    }

    protected function prepend_columns($table_name) {
        return [];
    }

    protected function prepend_indices($table_name) {
        return [];
    }

    protected function append_columns($table_name) {
        return [];    }

    protected function append_indices($table_name) {
        return [];
    }
}