<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/20/16
 * Time: 12:40 PM
 */

namespace Visionware\DataManager\Definition;

use Illuminate\Support\Collection;

class TableDefinition extends Definition {
    /**
     * @var SchemaDefinition
     */
    private $schema;
    /**
     * @var string
     */
    private $name;
    /**
     * @var Collection
     */
    private $fields;
    /**
     * @var Collection
     */
    private $indices;
    /**
     * @var Collection
     */
    public function __construct() {
        parent::__construct();
        $this->fields = new Collection;
        $this->indices = new Collection;
    }

    public function setSchema($schema) {
        $this->schema = $schema;
    }

    public function setName($name) {
        $this->name = $name;
    }

    public function putField($field) {
        $field->setTable($this);
        $this->fields->put($field->name(), $field);
    }

    public function putIndex($index) {
        $index->setTable($this);
        $this->indices->put($index->name(), $index);
    }

    /**
     * @return SchemaDefinition
     */
    public function schema() {
        return $this->schema;
    }

    public function name() {
        return $this->name;
    }

    /**
     * @return Collection|FieldDefinition[]
     */
    public function fields() {
        return $this->fields;
    }

    public function indices() {
        return $this->indices;
    }

    public function key() {
        return $this->getOther('key', []);
    }

    public function getPrimaryKeyType() {
        switch ($this->getOther('pk_type', 'int')) {
            case 'int':
                return 'int(11) unsigned';
            case 'uuid':
                return 'binary(16)';
            case 'none':
                return false;
        }
    }

    public function hasUuid() {
        return $this->getOther('pk_type', false) === 'uuid';
    }

    public function getPrimaryKeyExtra() {
        if ($this->getOther('pk_type', 'int') == 'int') return 'auto_increment';
        return '';
    }
    
    public function getPrimaryKeyField() {
        $field = FieldDefinition::hydrate([
            'name' => 'id',
            'type' => $this->getPrimaryKeyType(),
            'nullable' => false,
            'default' => '',
            'extra' => $this->getPrimaryKeyExtra()
        ]);
        $field->setTable($this);
        return $field;
    }

    public function getIngestFiles() {
        $files = $this->getOther('imported_from', []);
        if (!is_array($files)) $files = [$files];
        return $files;
    }

    public function getIsAppendOnly() {
        $append_only = $this->getOther('is_append_only', false);
        return $append_only;
    }

    public function getFieldColumnMap() {
        $map = [];
        foreach ($this->getImportFields() as $field) {
            $map[$field->getImportField()] = $field->name();
        }
        return $map;
    }

    /**
     * @return Collection|FieldDefinition[]
     */
    public function getImportFields() {
        return $this
            ->fields()
            ->filter(function ($item, $key) {
                return $item->hasImportField();
            })
        ;
    }

    public function isPartOfUniqueKey(FieldDefinition $field) {
        return in_array($field->name(), $this->key());
    }
    
    public function hasTimestamps() {
        if ($this->getOther('no_timestamps', false)) return false;
        return true;
    }

    public function isSyncedDown() {
        if ($this->getOther('is_synced_down', false)) return true;
        return false;
    }

    public function isSyncedUp() {
        if ($this->getOther('is_synced_up', false)) return true;
        return false;
    }

    public function isImported() {
        $hasImportedFromField = false;
        if ($this->fields()->filter(function ($item) { return $item->hasImportedFrom(); })->count() > 0) $hasImportedFromField = true;
        return ($this->getOther('imported_from', false)) !== false || $hasImportedFromField;
    }

    public function isInHistory() {
        return ($this->getOther('imported_from', false)) !== false;
    }

    public function hasDeleted() {
        if ($this->getOther('no_deleted', false)) return false;
        return true;
    }

    public function getSummarySourceTable() {
        foreach ($this->fields() as $field) {
            if ($field->importedFrom()) return $field->importedFrom();
        }
        return false;
    }

    public function isSummaryTable() {
        return $this->getOther('imported_from', false) == false;
    }

    public function getForeignDependentJoins() {
        $return = new Collection();
        foreach ($this->schema()->tables() as $table) {
            foreach ($table->fields() as $field) {
                /** @var FieldDefinition $field */
                if ($field->onTable() == $this->name()) {
                    if ($field->getOther('using') || $field->getOther('join_on')) {
                        if (!$field->table()->isSummaryTable()) {
                            $return->push($field);
                        }
                    }
                }
            }
        }
        return $return;
    }
//
//    public function getSourceTableName() {
//        if ($this->isSummaryTable()) {
//            foreach ($this->fields() as $field) {
//                if ($field->importedFrom()) return $field->importedFrom();
//            }
//        } else {
//            return $this->name() . '_latest';
//        }
//    }

    public function getJoins() {
        $joins = new Collection;
        $fieldsWithJoins = $this
            ->fields()
            ->filter(function($item) {
                /** @var FieldDefinition $item */
                return $item->hasJoin();
            })
        ;
        $joinNameSequenceLetter = "A";
        foreach ($fieldsWithJoins as $field) {
            /** @var FieldDefinition $field */
            foreach ($field->getJoins() as $joinArray) {
                $joinName = $joinArray['table'];
                while ($joins->has($joinName)) {
                    $joinName = $joinArray['table'] . $joinNameSequenceLetter++;
                }
                $joins->put($joinName, $joinArray);
            }
        }
        return $joins;
    }
    
    public static function hydrate($array) {
        $instance = new static();
        $instance->setName($array['table']);
        $instance->setOther(collect(array_except($array, ['table', 'fields', 'indices'])));
        foreach ($array['fields'] as $field) {
            $fieldDefinition = FieldDefinition::hydrate($field);
            $instance->putField($fieldDefinition);
        }
        foreach (array_get($array, 'indices', []) as $index) {
            $indexDefinition = IndexDefinition::hydrate($index);
            $instance->putIndex($indexDefinition);
        }
        return $instance;
    }

    public function validate() {
        foreach ($this->fields() as $field) $field->validate();
        foreach ($this->indices() as $index) $index->validate();
    }
}