<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/13/16
 * Time: 10:24 AM
 */

namespace Visionware\DataManager;

use Illuminate\Database\Connection;
use Log;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\SlackHandler;
use Monolog\Logger;
use PDOException;
use Storage;
use Config;
use DB;
use Symfony\Bridge\Monolog\Handler\ConsoleHandler;
use Symfony\Component\Console\Output\OutputInterface;
use Visionware\DataManager\Definition\SchemaDefinition;
use Visionware\DataManager\Exceptions\RemoteFileNotFoundException;
use Visionware\DataManager\Info\SchemaInfo;

class DataManagerHelper {

    protected $definition;
//    protected $history_db;
//    protected $live_db;

    public function __construct() {
        $this->load_definition();
//        try {
//
//        } catch (PDOException $e) {
//            Log::warning('Exception caught while connecting to history database', [$e]);
//            $this->history_db = null;
//        }
//        try {
//            $this->live_db = DB::connection(Config::get('datamanager.live-database-connection'));
//        } catch (PDOException $e) {
//            Log::warning('Exception caught while connecting to live database', [$e]);
//            $this->live_db = null;
//        }
    }

    /**
     * @return Ingester
     */
    public function getIngester() {
        return new Ingester($this->getSchemaInfo());
    }

    public function getTransferer() {
        return new Transferer($this->getSchemaInfo());
    }
//
    public function getImporter() {
        return new Importer($this->getSchemaInfo());
    }

    public function getUpdater() {
        return new Updater($this->getSchemaInfo());
    }

    /**
     * @param $mode
     * @return SchemaInfo
     */
    public function getSchemaInfo($mode = '') {
        $schemaclassname = 'Visionware\\DataManager\\Info\\' . ucfirst($mode) . 'SchemaInfo';
        $schemaInfo = $schemaclassname::createFromDefinition($this->getDefinition());
        return $schemaInfo;
    }

    public function getSqlSchemaInfo($mode, $infoClass = SchemaInfo::class) {
        $schemaInfo = $infoClass::createFromDefinition($this->getSqlDefinition($mode));
        return $schemaInfo;
    }

    public function getLogger($name, OutputInterface $output = null) {
        $logger = new Logger($name);
        $handler = new RotatingFileHandler(
            storage_path("/logs/datamanager/$name.log"),
            7,
            Logger::DEBUG,
            true,
            0666
        );
        $logger->pushHandler($handler);

        if (config('datamanager.slack.enabled')) {
            $slackHandler = new SlackHandler(config('datamanager.slack.token'), config('datamanager.slack.channel'), config('datamanager.slack.nick'), true, null, config('datamanager.slack.level', Logger::NOTICE), true, false, false);
            $logger->pushHandler($slackHandler);
        }
        
        if ($output) {
            $console_handler = new ConsoleHandler(
                $output, true, [
                    OutputInterface::VERBOSITY_NORMAL       => Logger::NOTICE,
                    OutputInterface::VERBOSITY_VERBOSE      => Logger::INFO,
                    OutputInterface::VERBOSITY_VERY_VERBOSE => Logger::INFO,
                    OutputInterface::VERBOSITY_DEBUG        => Logger::DEBUG,
                ]
            );
            $logger->pushHandler($console_handler);
        }

        return $logger;
    }

    /**
     * @return SchemaDefinition
     */
    public function getDefinition() {
        return SchemaDefinition::hydrate($this->definition);
    }

    public function getSqlDefinition($mode) {
        return SchemaDefinition::hydrate($this->createArrayFromConnection($this->getConnectionForMode($mode)));
    }

    public function getLiveConnection() {
        return DB::connection(Config::get('datamanager.live-database-connection'));
    }

    public function getImportConnection() {
        return DB::connection(Config::get('datamanager.import-database-connection'));
    }

    public function getHistoryConnection() {
        return DB::connection(Config::get('datamanager.history-database-connection'));
    }

    public function getHistoryImportConnection() {
        return DB::connection(Config::get('datamanager.historyimport-database-connection'));
    }

    public function getLiveDbName() {
        return Config::get('datamanager.live-database-schema');
    }

    public function getImportDbName() {
        return Config::get('datamanager.import-database-schema');
    }

    public function getHistoryDbName() {
        return Config::get('datamanager.history-database-schema');
    }

    public function getDbNameForMode($mode) {
        if ($mode == 'live') return $this->getLiveDbName();
        else if ($mode == 'history') return $this->getHistoryDbName();
        else if ($mode == 'import') return $this->getImportDbName();
        else if ($mode == 'historyimport') return $this->getImportDbName();
        return "";
    }

    public function getConnectionForMode($mode) {
        if ($mode == 'live') return $this->getLiveConnection();
        else if ($mode == 'history') return $this->getHistoryConnection();
        else if ($mode == 'historyimport') return $this->getHistoryImportConnection();
        else if ($mode == 'import') return $this->getImportConnection();
        return null;
    }
    
    public function getImportSourceDisk() {
        return Storage::disk(Config::get('datamanager.import_source.disk'));
    }
    
    public function getImportSourcePath() {
        return Config::get('datamanager.import_source.prepend_path');
    }

    public function getUpdaterFiles() {
        $files = Storage::disk(Config::get('datamanager.updater.disk'))->files(Config::get('datamanager.updater.path'));
        if ($files === false) return [];
        $arr = [];
        foreach ($files as $file) {
            $pi = pathinfo($file);
            if ($pi['extension'] != 'php') continue;
            $arr[basename($file)] = $file;
        }
        ksort($arr);
        return $arr;
    }

    private function load_definition() {
        $definition = [];
        $files = Storage::disk(Config::get('datamanager.disk'))->files(Config::get('datamanager.path'));
        $definition['tables'] = [];
        foreach ($files as $file) {
            $pi = pathinfo($file);
            if ($pi['extension'] != 'yml') continue;
            $yaml = yaml_parse_file($file);
            foreach ($yaml as $item) {
                if (array_key_exists('table', $item)) $definition['tables'][$item['table']] = $item;
            }
        }
        $this->definition = $definition;
    }

    private function createArrayFromConnection(Connection $connection) {
        $dbName = $connection->getDatabaseName();
        $result = $connection->select('select TABLE_NAME, COLUMN_NAME, COLUMN_TYPE, IS_NULLABLE, COLUMN_DEFAULT, EXTRA, CONSTRAINT_NAME, COLUMN_KEY, key_column_usage.ORDINAL_POSITION from information_schema.tables JOIN information_schema.columns USING (TABLE_SCHEMA, TABLE_NAME) LEFT JOIN information_schema.key_column_usage USING (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME) where tables.TABLE_SCHEMA = ? AND TABLE_TYPE = "BASE TABLE" GROUP BY TABLE_NAME, COLUMN_NAME ORDER BY TABLE_NAME, columns.ORDINAL_POSITION;', [$dbName]);

        $tables = [];
        foreach ($result as $row) {
            $tables[$row->TABLE_NAME]['table'] = $row->TABLE_NAME;
            $tables[$row->TABLE_NAME]['fields'][$row->COLUMN_NAME]['name'] = $row->COLUMN_NAME;
            $tables[$row->TABLE_NAME]['fields'][$row->COLUMN_NAME]['type'] = $row->COLUMN_TYPE;
            $tables[$row->TABLE_NAME]['fields'][$row->COLUMN_NAME]['nullable'] = $row->IS_NULLABLE == 'YES';
            $tables[$row->TABLE_NAME]['fields'][$row->COLUMN_NAME]['default'] = strtoupper((is_null($row->COLUMN_DEFAULT) && $row->IS_NULLABLE == 'YES') ? 'NULL' : $row->COLUMN_DEFAULT);
            $tables[$row->TABLE_NAME]['fields'][$row->COLUMN_NAME]['extra'] = strtoupper($row->EXTRA);
//            if ($row->COLUMN_KEY) {
//                $type = 'UNKNOWN';
//                if ($row->COLUMN_KEY == 'PRI') $type = 'PRIMARY KEY';
//                else if ($row->COLUMN_KEY == 'MUL') $type = 'KEY';
//                else if ($row->COLUMN_KEY == 'UNI') $type = 'UNIQUE KEY';
//                $position = is_null($row->ORDINAL_POSITION) ? 0 : $row->ORDINAL_POSITION;
//                $name = is_null($row->CONSTRAINT_NAME) ? $row->COLUMN_NAME : $row->CONSTRAINT_NAME;
//                $tables[$row->TABLE_NAME]['indices'][$name]['name'] = $name;
//                $tables[$row->TABLE_NAME]['indices'][$name]['type'] = $type;
//                $tables[$row->TABLE_NAME]['indices'][$name]['columns'][$position] = $row->COLUMN_NAME;
//            }
        }
        foreach ($tables as $tableArray) {
            $table = $tableArray['table'];
            $result = $connection->select("SHOW INDEXES FROM $table");
            foreach ($result as $row) {
                $type = 'KEY';
                if ($row->Key_name == 'PRIMARY') $type = 'PRIMARY KEY';
                else if ($row->Non_unique == '0') $type = 'UNIQUE KEY';
                $tables[$table]['indices'][$row->Key_name]['type'] = $type;
                $tables[$table]['indices'][$row->Key_name]['columns'][] = $row->Column_name;
                $tables[$table]['indices'][$row->Key_name]['name'] = $row->Key_name;
            }
        }
        
        $views = [];

        $result = $connection->select('select TABLE_NAME, VIEW_DEFINITION from information_schema.views where TABLE_SCHEMA = ?;', [$dbName]);
        foreach ($result as $row) {
            $replaceString = "`$dbName`.";
            $views[$row->TABLE_NAME]['name'] = $row->TABLE_NAME;
            $views[$row->TABLE_NAME]['statement'] = str_replace(['select ', ' from ', $replaceString], ['SELECT ', ' FROM ', ""], $row->VIEW_DEFINITION);
        }
        return ['tables' => $tables, 'views' => $views];
    }

    /** THIS IS NO LONGER USED, USE THE datamanager:update COMMAND INSTEAD */
//    public static function getPostImportSql() {
//        $files = Storage::disk(Config::get('datamanager.disk'))->files(Config::get('datamanager.postImportSqlPath'));
//        sort($files);
//        $raw = '';
//        foreach ($files as $file) {
//            $raw .= "\n" . file_get_contents($file);
//        }
//        $raw = explode("\n", $raw);
//        $sql = [];
//        foreach ($raw as $line) {
//            $line = trim($line);
//            $line = str_replace("\t", " ", $line);
//            if (!strlen($line)) continue;
//            if (starts_with($line, '#')) continue;
//            if (starts_with($line, '--')) continue;
//            if (starts_with($line, 'START TRANSACTION')) continue;
//            if (starts_with($line, 'COMMIT')) continue;
//            $sql[] = "$line ";
//        }
//        $sql = array_map('trim', explode(';', implode('', $sql)));
//        foreach ($sql as $index=>$item) {
//            if (!strlen($item)) unset($sql[$index]);
//        }
////        $sql = array_map('trim', explode(';', str_replace(["\n", "\t"], " ", $sql)));
//        return $sql;
//    }

    public static function createImportMutex() {
        touch(self::importMutexPath());
    }

    public static function deleteImportMutex() {
        unlink(self::importMutexPath());
    }

    public static function isImportRunning() {
        return file_exists(self::importMutexPath());
    }

    public static function importMutexPath() {
        return storage_path('import_is_running');
    }

    public static function acquireImportLock() {
        $lockFile = fopen(config('datamanager.importer.lock_path'), 'w');
        flock($lockFile, LOCK_EX);
    }

    public static function releaseImportLock() {
        $lockFile = fopen(config('datamanager.importer.lock_path'), 'w');
        flock($lockFile, LOCK_UN);
    }

//    private function sql_indices($table_name) {
//        $return = [];
//        $indices = $this->connection->select("SHOW INDEXES FROM $table_name");
//        foreach ($indices as $index_col) {
//            if ($index_col->Key_name == 'PRIMARY') $type = 'PRIMARY KEY';
//            else if ($index_col->Non_unique == '0') $type = 'UNIQUE KEY';
//            else $type = 'KEY';
//
//            $return[$index_col->Key_name]['type'] = $type;
//            $return[$index_col->Key_name]['columns'][] = $index_col->Column_name;
//            $return[$index_col->Key_name]['name'] = ($index_col->Key_name == 'PRIMARY') ? '' : $index_col->Key_name;
//        }
//        foreach ($return as $name=>$item) {
//            if (!strlen($item['name'])) $return[$name]['name'] = implode('-', $item['columns']);
//        }
//        if (array_key_exists('PRIMARY', $return)) {
//            $keys = array_keys($return);
//            $keys[array_search('PRIMARY', $keys)] = $return['PRIMARY']['name'];
//            $return = array_combine($keys, $return);
//        }
//        return $return;
//    }
}