<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/13/16
 * Time: 10:16 AM
 */

namespace Visionware\DataManager;

use Visionware\DataManager\Definition\FieldDefinition;
use Visionware\DataManager\Facades\DataManager;
use Carbon\Carbon;
use Illuminate\Database\Connection;
use Visionware\DataManager\Exceptions\IngestionFieldsMismatchException;
use Visionware\DataManager\Exceptions\RemoteFileDownloadException;
use Visionware\DataManager\Exceptions\RemoteFileNotFoundException;
use Visionware\DataManager\Exceptions\UnableToCreateLoadFileException;

class Importer extends DataManagerProcess {
    /**
     * @var Connection
     */
    protected $historyDb;
    /**
     * @var Connection
     */
    protected $liveDb;
    private $iterations;
    private $sourceTableName;
    private $liveTableName;
    private $historySchemaName;
    private $liveSchemaName;
    private $isSummary;

    public function __construct($schema) {
        parent::__construct($schema);
        $this->historyDb = DataManager::getHistoryConnection();
        $this->liveDb = DataManager::getLiveConnection();
        $this->historySchemaName = DataManager::getHistoryDbName();
        $this->liveSchemaName = DataManager::getLiveDbName();
    }

    public function go() {
        try {
//            $this->live_db->beginTransaction();
            foreach ($this->schema->tablesToImport() as $tableName) {
                $this->setTable($tableName);
                $start = Carbon::now();
                $this->notice("Starting to import table " . $tableName);
                $this->importTable();
                $diff = Carbon::now()->diffForHumans($start, true);
                $this->notice("Finished importing table " . $tableName . " in $diff total");
            }
//            $this->live_db->commit();
        } catch (\Exception $e) {
            $this->error("Caught exception while importing table {$tableName}! Rolling back...", ['exception' => $e]);
            $this->liveDb->rollBack();
        }
    }

    public function importTable() {
        $this->isSummary = $this->definition->isSummaryTable();

        $this->sourceTableName = $this->historySchemaName . '.' . $this->definition->getSourceTableName() . '_latest';
        $this->liveTableName = $this->liveSchemaName . '.' . $this->tableName;

        $sourceLastModified = $this->getSourceLastModified();
        $lastUpdated = $this->getLiveLastUpdated();
        $isModified = !($lastUpdated >= $sourceLastModified);

        if (!$isModified && !$this->force) {
            $this->info("Table {$this->tableName} is up-to-date, skipping");
            return;
        }

        $this->iterations = 1;

        $foreigners = $this->foreigners();

        $joinsResult = $this->joins($foreigners);
        $joins = $joinsResult['joins'];
        $tableNameMapping = $joinsResult['table_name_mapping'];

        $columns = $this->buildColumns($tableNameMapping);

        $this->info(sprintf("Updating live table %s (%d %s)...", $this->liveTableName, $this->iterations, str_plural('iteration', $this->iterations)));
        $deletedJoinsString = $this->buildDeletedJoinsString();
        $this->updateLive($columns, $joins, $deletedJoinsString);

        $this->info('Updating last modified metadata record...');
        $this->updateMetadata($sourceLastModified);


    }

    private function getSourceLastModified() {
        return $this->liveDb->table($this->historyDb->getDatabaseName() . '.' . $this->definition->getSourceTableName() . '_files')->max('last_modified');
    }

    private function getLiveLastUpdated() {
        return $this->liveDb->table('datamanager_metadata')
            ->where('metadata_type', '=', 'last_updated')
            ->where('metadata_key', '=', $this->tableName)
            ->value('date_value')
        ;
    }

    private function foreigners() {
        $foreigners = [];
        foreach ($this->definition->fields() as $field) {
            $letter = "A";
            foreach ($field->getExtraJoins() as $extraJoin) {
                $fake = ['name' => $field->name(), 'type' => $field->type()];
                if (array_key_exists('imported_from', $extraJoin)) $fake['imported_from'] = $extraJoin['imported_from'];
                if (array_key_exists('on_table', $extraJoin)) $fake['on_table'] = $extraJoin['on_table'];
                if (array_key_exists('using', $extraJoin)) $fake['using'] = $extraJoin['using'];
                if (array_key_exists('join_on', $extraJoin)) $fake['join_on'] = $extraJoin['join_on'];
                $foreigners[$field->name() . $letter++] = FieldDefinition::hydrate($fake);
            }
            if ($field->hasForeign()) {
                $foreigners[$field->name()] = $field;
                if ($field->onTable() == $this->tableName) $this->iterations++;
            }
        }
        return $foreigners;
    }

    private function joins($foreigners) {
        if ($this->isSummary) {
            return $this->summaryJoins($foreigners);
        } else {
            return $this->importJoins($foreigners);
        }
    }

    private function importJoins($foreigners) {
        $uniqueKeyColumnsString = implode(', ', $this->definition->key());
        $joins = [];
        $joins[$this->tableName] = "LEFT OUTER JOIN {$this->liveTableName} USING ($uniqueKeyColumnsString)";
        $tableNameMapping = [];
        $letter = "A";
        /** @var FieldDefinition $field */
        foreach ($foreigners as $field) {
            $foreignTableBase = "{$this->liveSchemaName}.{$field->onTable()}";
            $foreignTable = $field->onTable();
            while (array_key_exists($foreignTable, $joins)) {
                $foreignTable = $field->onTable() . $letter++;
            }
            $joinOns = [];
            foreach ($field->joinOns() as $importJoinColumn) {
                $joinOns[] = $foreignTable
                    . '.'
                    . $importJoinColumn['foreign']
                    . ' = '
                    . $this->sourceTableName
                    . '.'
                    . $importJoinColumn['local'];
                $tableNameMapping[$field->name()] = $foreignTable;
            }
            $joinOnsString = implode(' AND ', $joinOns);

            $joins[$foreignTable] = "LEFT OUTER JOIN $foreignTableBase $foreignTable ON ($joinOnsString)";
        }
        return ['joins' => $joins, 'table_name_mapping' => $tableNameMapping];
    }

    private function summaryJoins($foreigners) {
        $uniqueKeyColumnsString = implode(', ', $this->definition->key());

        $joins = [];
        $joins[$this->tableName] = "LEFT OUTER JOIN {$this->liveTableName} USING ($uniqueKeyColumnsString)";
        $tableNameMapping = [];
        /** @var FieldDefinition $field */
        foreach ($foreigners as $field) {
            if ($field->onTable()) {
                $foreignTableBase = "{$this->liveSchemaName}.{$field->onTable()}";
                $foreignTable = $field->onTable();
            } else if ($field->importedFrom()) {
                $foreignTableBase = "{$this->historySchemaName}.{$field->importedFrom()}_latest";
                $foreignTable = $field->importedFrom();
            }
            if ($field->hasJoinOns()) {
                $join_ons = [];
                foreach ($field->joinOns() as $importJoinColumn) {
                    $join_ons[] = $foreignTable
                        . '.'
                        . $importJoinColumn['foreign']
                        . ' = '
                        . $this->sourceTableName
                        . '.'
                        . $importJoinColumn['local'];
                }
                $joinOnsString = implode(' AND ', $join_ons);

                $joins[$foreignTable] = "LEFT JOIN $foreignTableBase $foreignTable ON ($joinOnsString)";
            } else if ($field->hasUsing()) {
                $joins[$foreignTable] = "LEFT JOIN $foreignTableBase $foreignTable USING ({$field->using()})";
            }
            $tableNameMapping[$field->name()] = $foreignTable;
        }
        return ['joins' => $joins, 'table_name_mapping' => $tableNameMapping];
    }

    private function buildColumns($tableNameMapping) {
        $liveColumns = [];
        $selectColumns = [];
        if ($this->definition->hasUuid()) {
            $liveColumns = ['id'];
            $selectColumns[] = "COALESCE({$this->liveTableName}.id, UuidToBin(UUID()))";
        }
        $liveUpdates = [];
        foreach ($this->definition->fields() as $field) {
            if ($field->skipImport()) continue;
            $columnKey = $field->name();
            if ($field->hasForeign()) {
                $selectColumns[] = $tableNameMapping[$columnKey] . ".{$field->foreign()} {$field->name()}";
            } else {
//                if ($this->isSummary) {
                    $selectColumns[] = $this->sourceTableName . '.' . $field->name();
//                } else {
//                    $selectColumns[] = $field->getTransformation(true);
//                }
            }

            $liveColumns[] = $field->name();
            $liveUpdates[] = "{$field->name()}=VALUES({$field->name()})";
        }
        $liveUpdates[] = "deleted_at=NULL";
        return [
            'live' => $liveColumns,
            'select' => $selectColumns,
            'update' => $liveUpdates,
        ];
    }

    private function updateLive($columns, $joins, $deletedJoinsString) {
        $liveColumns = $columns['live'];
        $selectColumns = $columns['select'];
        $liveUpdates = $columns['update'];

        $selectColumnsString = implode(', ', $selectColumns);
        $liveColumnsString = implode(', ', $liveColumns);
        $liveUpdatesString = implode(', ', $liveUpdates);

        $joinsString = implode(" ", $joins);
        $count = 1;
        while ($this->iterations > 0) {
            $this->info("Inserting into live (iteration $count)...");
            $this->insertLive($liveColumnsString, $selectColumnsString, $joinsString, $liveUpdatesString);

            $this->info("Updating deleted records (iteration $count)...");
            $this->updateDeleted($deletedJoinsString);
            $this->iterations--;
            $count++;
        }
    }

    private function insertLive($liveColumnsString, $selectColumnsString, $joinsString, $liveUpdatesString) {
        $sql =
            "INSERT IGNORE INTO {$this->liveTableName} ($liveColumnsString) SELECT $selectColumnsString FROM {$this->sourceTableName} $joinsString ON DUPLICATE KEY UPDATE $liveUpdatesString;";
        $this->debug($sql);
        $this->liveDb->affectingStatement($sql);
    }

    private function buildDeletedJoinsString() {
        $joins = [];
        foreach ($this->definition->key() as $keyColumn) {
//            $liveColumn = $this->table_field($keyColumn);
            $joins[] = $this->liveTableName . '.' . $keyColumn . ' = ' . $this->sourceTableName . '.' . $keyColumn;
        }
        return implode(' AND ', $joins);
    }

    private function updateDeleted($deletedJoinsString) {
        $sql =
            "UPDATE {$this->liveTableName} LEFT OUTER JOIN {$this->sourceTableName} ON ($deletedJoinsString) SET {$this->liveTableName}.deleted_at = NOW() WHERE {$this->sourceTableName}.record_hash IS NULL AND {$this->liveTableName}.deleted_at IS NULL;";
        $this->debug($sql);
        $this->liveDb->affectingStatement($sql);
    }

    private function updateMetadata($sourceLastModified) {
        $sql =
            "REPLACE INTO datamanager_metadata (`metadata_type`, `metadata_key`, `date_value`) VALUES ('last_updated', '{$this->tableName}', '$sourceLastModified')";
        $this->debug($sql);
        $this->liveDb->affectingStatement($sql);
    }
    
    private function copyFromHistoryToLive() {
        $tableInfo = DataManager::getSchemaInfo('live')->tables()->get('Divisions');
        dd($tableInfo);

        $qualifiedLiveTable = $tableInfo->getQualifiedName();
        
//        $sql = $this->compileInsertStatement()
    }

    private function compileInsertStatement($qualifiedLiveTable, $insertColumnsString, $selectColumnsString, $qualifiedHistoryTable, $joinString, $onDuplicateKeyString) {
        $sql = <<<SQLEND
INSERT IGNORE INTO $qualifiedLiveTable ($insertColumnsString)
SELECT ($selectColumnsString) FROM $qualifiedHistoryTable
{$joinString}
ON DUPLICATE KEY UPDATE $onDuplicateKeyString
SQLEND;
        return $sql;
    }
}