<?php

namespace Visionware\DataManager\Actions;

use Closure;
use Illuminate\Support\Collection;
use Visionware\DataManager\Definition\Definition;
use Visionware\DataManager\Definition\SchemaDefinition;
use Visionware\DataManager\Definition\TableDefinition;
use Visionware\DataManager\Info\ColumnInfo;
use Visionware\DataManager\Info\IndexInfo;
use Visionware\DataManager\Info\Info;
use Visionware\DataManager\Info\SchemaInfo;
use Visionware\DataManager\Info\TableInfo;
use Visionware\DataManager\Info\ViewInfo;

/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/21/16
 * Time: 10:14 AM
 * @method prependColumn(Closure $callback) Prepend a column to the table. Callback arguments: TableDefinition, TableInfo. Callback should return ColumnInfo or a Collection of ColumnInfos
 * @method appendColumn(Closure $callback) Append a column to the table. Callback arguments: TableDefinition, TableInfo. Callback should return ColumnInfo or a Collection of ColumnInfos
 * @method skipColumn(Closure $callback) Skip a column that is already defined in the table. Callback arguments: TableDefinition, TableInfo. Callback should return ColumnInfo or a Collection of ColumnInfos
 * @method prependIndex(Closure $callback) Prepend an index to the table. Callback arguments: TableDefinition, TableInfo. Callback should return IndexInfo or a Collection of IndexInfos
 * @method appendIndex(Closure $callback) Append an index to the table. Callback arguments: TableDefinition, TableInfo. Callback should return IndexInfo or a Collection of IndexInfos
 * @method siblingTable(Closure $callback) Append a table to the schema after this one. Callback arguments: TableDefinition, TableInfo. Callback should return TableInfo or a Collection of TableInfos
 * @method replaceTable(Closure $callback) Replace a table with a new one (or several new ones). Callback arguments: TableDefinition, TableInfo. Callback should return TableInfo or a Collection of TableInfos
 * @method viewFromTable(Closure $callback) Prepend a column to the table. Callback arguments: TableDefinition, TableInfo. Callback should return ViewInfo or a Collection of ViewInfos
 * @method skipTable(Closure $callback) Skip a table that is already defined in the schema. Callback arguments: SchemaDefinition, SchemaInfo. Callback should return TableInfo or a Collection of TableInfos
 */
class TableAction extends Action {
    protected $allowedOperations = [
        'prependColumn',
        'appendColumn',
        'skipColumn',
        'prependIndex',
        'appendIndex',
        'siblingTable',
        'replaceTable',
        'viewFromTable',
        'skipTable',
    ];

    protected function executeAppendColumn(ColumnInfo $newColumn, TableInfo $tableInfo) {
        $tableInfo->putColumn($newColumn);
        return $tableInfo;
    }

    protected function executePrependColumn(ColumnInfo $newColumn, TableInfo $tableInfo) {
        $tableInfo->prependColumn($newColumn);
        return $tableInfo;
    }
    
    protected function executeSkipColumn(ColumnInfo $skipColumn, TableInfo $tableInfo) {
        $tableInfo->deleteColumn($skipColumn);
        return $tableInfo;
    }

    protected function executeAppendIndex(IndexInfo $newIndex, TableInfo $tableInfo) {
        $tableInfo->putIndex($newIndex);
        return $tableInfo;
    }

    protected function executePrependIndex(IndexInfo $newIndex, TableInfo $tableInfo) {
        $tableInfo->prependIndex($newIndex);
        return $tableInfo;
    }

    protected function executeSiblingTable(TableInfo $siblingTable, TableInfo $tableInfo) {
        $siblingTable->setDerived();
        $tableInfo->schema()->putTable($siblingTable);
        return $tableInfo;
    }

    protected function executeReplaceTable(TableInfo $replaceTable, TableInfo $tableInfo) {
        $replaceTable->setDerived();
        $tableInfo->schema()->putTable($replaceTable);
        $tableInfo->schema()->removeTable($tableInfo);
        return $tableInfo;
    }

    protected function executeViewFromTable(ViewInfo $viewInfo, TableInfo $tableInfo) {
        $tableInfo->schema()->putView($viewInfo);
        return $tableInfo;
    }

    protected function executeSkipTable(bool $skip, TableInfo $tableInfo) {
        if ($skip) $tableInfo->schema()->removeTable($tableInfo);
        return $tableInfo;
    }
}